#ifndef rl_MultilayerSurface_h_INCLUDED
#define rl_MultilayerSurface_h_INCLUDED

// File:   rl_MultilayerSurface.h
// Author: Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006, 2007 Smithsonian Astrophysical Observatory
 *
 * This file is part of rl_raylib
 *
 * rl_raylib is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * rl_raylib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <dvm3/dvm3_vector.h>               // dvm3_Vector
#include <rl_raylib/rl_Multilayer.h>        // rl_Multilayer
#include <rl_raylib/rl_Ray.h>               // rl_Ray
#include <rl_raylib/rl_ReflectionCoefPOD.h> // rl_ReflectionCoefPOD

class rl_DielectricLayer;                   // forward declaration

//########################################################################
// rl_MultilayerSurface
//########################################################################
//
/** 
 * \class rl_MultilayerSurface
 *
 * A class encapsulating a multilayer surface, including surface normal.
 * This class contains the multilayer data and surface normal
 * \em by \em reference.  That is, the rl_Multilayer
 * and dvm3_Vector objects  must already exist.
 */

class rl_MultilayerSurface
{
private:

  /// multilayer information
  rl_Multilayer&        ml_;
  /// surface normal
  dvm3_Vector&          normal_;

  /// multilayer reflectance (cached)
  rl_ReflectionCoefPOD  rflcoef_;

  /// PROHIBITED method; declared private and not defined.
  rl_MultilayerSurface();
  /// PROHIBITED method; declared private and not defined.
  rl_MultilayerSurface( rl_MultilayerSurface const& );
  /// PROHIBITED method; declared private and not defined.
  rl_MultilayerSurface& operator=( rl_MultilayerSurface const& );

public:

  /**
   * Destructor.
   */
  virtual ~rl_MultilayerSurface( );

  /**
   * Constructor.  Constructs multilayer surface information from 
   * the provided rl_Multilayer object and normal vector.
   *
   * @param ml the rl_Multilayer object to be used (must already exist)
   * @param norm the surface normal dvm3_Vector to be used (must already exist)
   */
  rl_MultilayerSurface( rl_Multilayer& ml, dvm3_Vector& norm );

  /**
   * Accessor.
   *
   * @return the surface normal vector at the ray intercept
   */
  dvm3_Vector const& normal_vector() const;

  /**
   * Mutator. Set the surface normal at the ray/surface intercept.  
   * Until the full surface intercept apparatus is available, 
   * applications need to set the surface normal at the intercept.
   *
   * @param norm the surface normal vector at the ray intercept
   */
  void set_normal( dvm3_Vector const& norm );

  /*
   * Accessor.  Find ray intercept with this surface (PLACEHOLDER)
   *
   * @return distance to the intercept along the ray.
   *
   * @param ray for which the surface intercept is to be found.
   *
   * @return 0 if successful.
   */
  // int surface_intercept( rl_Ray& ray );

  /**
   * Mutator.  Reflect ray direction vector at this surface
   *
   * @param ray to be reflected.  The ray is assumed to be
   *        already at a valid surface intercept point.  The ray
   *        direction vector and polarization state are updated.
   *
   * @return 0 if successful.
   */
  int reflect( rl_Ray& ray );

  /*
   * Mutator.  Apply scatter to a ray.  (PLACEHOLDER)
   *
   * @param ray to be scattered.  The ray is assumed to be
   *        already at a valid surface intercept point.  The ray
   *        direction vector and polarization state are updated.
   */
  // int scatter( rl_Ray& ray );

  /** 
   * Accessor.
   * 
   * @return rl_ReflectionCoefPOD holding the complex perpendicular and
   *         parallel reflection coefficients.
   */
  rl_ReflectionCoefPOD const& reflection_coefs() const;

  /** 
   * Accessor.
   * 
   * @param layer_no layer number for which the information is requested.
   *
   * @return const reference to the rl_DielectricLayer for layer layer_no.
   */
  rl_DielectricLayer const& layer( int layer_no ) const;

  /**
   * Dump information about a layer.
   *
   * @param os       output stream
   * @param layer_no layer number for which the information is requested.
   * @param pre optional char* prefix string.
   * @param pst optional char* postfix string.
   */
  std::ostream& dump_on( std::ostream& os, int layer_no,
                                           char const pre[] = "",
                                           char const pst[] = "" ) const;
};

//########################################################################
//########################################################################
//
//    #    #    #  #          #    #    #  ######   ####
//    #    ##   #  #          #    ##   #  #       #
//    #    # #  #  #          #    # #  #  #####    ####
//    #    #  # #  #          #    #  # #  #            #
//    #    #   ##  #          #    #   ##  #       #    #
//    #    #    #  ######     #    #    #  ######   ####
//
//########################################################################
//########################################################################

//=========================================================================
// dtor; ctors

//-------------------------------------------------------------------------
inline rl_MultilayerSurface::
rl_MultilayerSurface( rl_Multilayer& ml, dvm3_Vector& norm )
  : ml_( ml ), normal_( norm )
{}

//=========================================================================
// accessors

//-------------------------------------------------------------------------
inline dvm3_Vector const& rl_MultilayerSurface::
normal_vector() const
{ return normal_; }

//-------------------------------------------------------------------------
inline rl_ReflectionCoefPOD const& rl_MultilayerSurface::
reflection_coefs() const
{ return rflcoef_; }
  
//=========================================================================
// mutators

//-------------------------------------------------------------------------
inline void rl_MultilayerSurface::
set_normal( dvm3_Vector const& norm )
{ normal_ = norm; }
  
// //-------------------------------------------------------------------------
// inline double rl_MultilayerSurface::
// intercept( rl_Ray& ray )
// { return 0.0; }

//-------------------------------------------------------------------------
inline int rl_MultilayerSurface::
reflect( rl_Ray& ray )
{
  int rc = ml_.multilayer_reflect_coef( rflcoef_, ray.energy(), 
                                        dot( ray.direction(), normal_ ) );
  if (! rc )
  {
    ray.reflect( normal_, rflcoef_ );
  }
  return rc;
}

//-------------------------------------------------------------------------
inline rl_DielectricLayer const& rl_MultilayerSurface::
layer( int layer_no ) const
{ return ml_.layer(layer_no); }

//-------------------------------------------------------------------------
inline std::ostream& rl_MultilayerSurface::
dump_on( std::ostream& os, int  layer_no,
                           char const pre[],
                           char const pst[] ) const
{ 
  ml_.dump_on( os, layer_no, pre, pst ); 
  return os;
}

// //-------------------------------------------------------------------------
// inline void rl_MultilayerSurface::
// scatter( rl_Ray& ray )
// {}

// rl_MultilayerSurface_h_INCLUDED
#endif
