#ifndef rl_DielectricData_h_INCLUDED
#define rl_DielectricData_h_INCLUDED

// File:   rl_DielectricData.h
// Author: Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006, 2007 Smithsonian Astrophysical Observatory
 *
 * This file is part of rl_raylib
 *
 * rl_raylib is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * rl_raylib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

/****************************************************************************
 * Description: declare ReflectanceInfo, a descriptor for the reflectance
 *              information
 *
 * Note       : (1-alpha) is real part of complex dielectric constant
 *              (-gamma)  is imag part of complex dielectric constant
 *
 * History
 *--------
 * 1.0.1 2004-Dec-07  tjg  simplify/remove include guards
 * 1.0.0 1998-Jun-12  tjg  fold rflinfo_private.h back into rflinfo.h
 * 1.0.0 1998-Jun-12  tjg  fold rflinfo_private.h back into rflinfo.h
 * 0.0.0 1995-May-18  tjg  original version
 */

#include <cstdlib>                    // size_t
#include <cstring>                    // memcpy
#include <cfloat>                     // DBL_MAX
#include <rl_raylib/rl_Traits.h>      // rl_Traits::EInterpMode
#include <rl_raylib/rl_Exception.h>   // rl_Exception

// #define TEST_ALPHA_GAMMA

//########################################################################
// rl_DielectricData
//########################################################################
//
/** 
 * \class rl_DielectricData
 * A class encapsulating the dielectric data (alpha, gamma) as a 
 * function of energy.
 */

class rl_DielectricData
{
public:

  /** 
   * \struct rl_DielectricData::rl_DielectricDataBinPOD
   *
   * A POD describing the lower and upper edge of an energy bin.
   */
  struct rl_DielectricDataBinPOD
  {
    /// lower edge of energy bin
    rl_Traits::rl_DielectricPOD lo_;
    /// upper edge of energy bin
    rl_Traits::rl_DielectricPOD hi_;
  };

private:

  /// number of elements in the rl_DielectricData internal array
  size_t                    nelts_;
  /// array of dielectric decrements
  rl_DielectricDataBinPOD*  decrement_;    // dielectric data
  /// minimum energy covered by the dielectric data
  double                    emin_;
  /// maximum energy covered by the dielectric data
  double                    emax_;
  /// bulk density for the dielectric data
  double                    bulkdensity_;
  /// interpolation mode for the dielectric data
  rl_Traits::EInterpMode    interp_mode_;

  /// assignment is prohibited
  rl_DielectricData& operator=( rl_DielectricData const& ); // PROHIBITED

public:

  /** 
   * Non-virtual detructor.
   */
 ~rl_DielectricData();

  /** 
   * Default constructor.  Constructs a vacuum rl_DielectricData.
   * Use init to initialize the object to a state other than vacuum.
   */
  rl_DielectricData();

  /** 
   * Copy constructor
   * @param other rl_DielectricData to be copied.
   */
  rl_DielectricData( rl_DielectricData const& other );

  /** 
   * Constructor.  The data for the dielectric constants are obtained 
   * from a c-style array of const rl_DielectricPOD; the array contains 
   * num_pts points.
   * The interpolation mode will be set to interp_mode.
   * The optical constants will be scaled by bulk_density.
   * @param diel an array of <code>num_pts</code> POD's containing
   *           the complex dielectric decrements.
   * @param num_pts number of elements in diel.
   * @param interp_mode type of interpolation to be used in interpolating
   *           the dielectric constants.
   * @param bulk_density relative bulk density factor to be used; 1.0
   *           for full bulk density.  The dielectric decrements will be 
   *           scaled by bulk_density.
   */
  rl_DielectricData( rl_Traits::rl_DielectricPOD const* diel, 
                     size_t                             num_pts,
                     rl_Traits::EInterpMode             interp_mode,
                     double                             bulk_density = 1.0 );

  /**
   * Initialization function.
   * The data are obtained from a c-style array of const
   * rl_DielectricPOD; the array contains num_pts points.
   * The interpolation mode will be set to interp_mode.
   * The optical constants will be scaled by bulk_density.
   * @param diel an array of <code>num_pts</code> POD's containing
   *        the complex dielectric decrements.
   * @param num_pts number of elements in diel.
   * @param interp_mode type of interpolation to be used in interpolating
   *        the dielectric constants.
   * @param bulk_density relative bulk density factor to be used; 1.0
   *        for full bulk density.  The dielectric decrements will be 
   *        scaled by bulk_density.
   */
  void init( rl_Traits::rl_DielectricPOD const* diel, 
             size_t                             num_pts,
             rl_Traits::EInterpMode             interp_mode,
             double                             bulk_density = 1.0 );

  /**
   * Evaluate the dielectric decrements, alpha and gamma, at the given energy.
   * The energy must be between the minimum and the maximum energies
   * in the <code>rl_DielectricPOD</code> array used to
   * initialize the <code>rl_DielectricData</code>
   * Returns: 0 on success; -1 if energy is below lower limit,
   * +1 if energy is above upper limit.
   * @param energy photon energy (in keV).
   * @param alpha real part of the complex dielectric decrement.
   * @param gamma imaginary part of the complex dielectric decrement.
   */
  int alpha_gamma( double energy, double& alpha, double& gamma );

  /**
   * Return the minimum energy covered by this dataset.
   */
  double energy_min() const;

  /**
   * Return the maximum energy covered by this dataset.
   */
  double energy_max() const;

  /**
   * Return the maximum energy covered by this dataset.
   * Return the  relative bulk density for this set of optical constants.
   * 1.0 is nominal full bulk density
   */
  double bulk_density_factor() const;

  /**
   * Returns True if the object is a vacuum state; False, otherwise.
   */
  rl_Traits::Bool is_vacuum() const;
};

//########################################################################
//########################################################################
//
//    #    #    #  #          #    #    #  ######   ####
//    #    ##   #  #          #    ##   #  #       #
//    #    # #  #  #          #    # #  #  #####    ####
//    #    #  # #  #          #    #  # #  #            #
//    #    #   ##  #          #    #   ##  #       #    #
//    #    #    #  ######     #    #    #  ######   ####
//
//########################################################################
//########################################################################

//=========================================================================
// ctors...

//-------------------------------------------------------------------------
inline rl_DielectricData::
rl_DielectricData()
  : nelts_(0)
  , decrement_(0)
  , emin_(0.0)
  , emax_(DBL_MAX)
  , bulkdensity_(1.0)
  , interp_mode_(rl_Traits::ELinLin)
{}

//=========================================================================
// accessors...

//-------------------------------------------------------------------------
inline double rl_DielectricData::
energy_min() const
{ return emin_; }

//-------------------------------------------------------------------------
inline double rl_DielectricData::
energy_max() const
{ return emax_; }

//-------------------------------------------------------------------------
inline double rl_DielectricData::
bulk_density_factor() const
{ return bulkdensity_; }

//-------------------------------------------------------------------------
inline rl_Traits::Bool rl_DielectricData::
is_vacuum() const
{ return rl_Traits::Bool(decrement_ == 0); }

// rl_DielectricData_h_INCLUDED
#endif
