#ifndef dvm3_vector_h_INCLUDED
#define dvm3_vector_h_INCLUDED

// File:    dvm3_vector.h
// Author:  Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of dvm3
 *
 * dvm3 is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * dvm3.cvs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

/****************************************************************************
 * Description: collection of methods for dvm3_Vector
 */

#include <cfloat>                       // DBL_EPSILON
#include <vm_math/vm_v3math.h>          // vm_V3Math<T_fp>
#include <iosfwd>
#include <cstdio>

//########################################################################
// dvm3_VPOD
//########################################################################

/*@{*/
/** 
 * \internal
 *
 * \struct dvm3_VPod dvm3_vector.h <dvm3/dvm3_vector.h>
 *
 * dvm3_VPOD is a Plain Ol' Data base class to represent the vector data.
 * dvm3_VPOD is declared as a struct encapsulating an array of doubles
 * in order to assure that the data alignment and contiguity are maintained.
 * This implementation is exposed purely for efficiency considerations; users
 * shall not assume that this implementation detail will not change in future.
 * DO NOT rely on the fact that there is an array under the hood or the
 * naming of the data members within dvm3_VPOD.
 *
 * NOTE:  a POD class must satisfy (Jack W. Reeves, C++Report, February 1998,
 *         p. 46-54):
 *  - no user-declared constructors, copy-assign operator, or destructor
 *  - private or protected nonstatic data members
 *  - base classes
 *  - virtual functions
 *  - nonstatic data members of type pointer to member,
 *     non-POD-struct/class (or array of such types) or reference
 */
struct dvm3_VPOD
{
  double x_[3];
};
/*@}*/

//########################################################################
// dvm3_Vector
//########################################################################

/* forward declaration required below */
class dvm3_Matrix;

/*!
 * \addtogroup dvm3_vector
 */
/** 
 * \class dvm3_Vector dvm3_vector.h <dvm3/dvm3_vector.h>
 *
 * A class representing a 3-vector of doubles with common numerical 
 * operations defined.  The dvm3_Vector class is a simple class to 
 * handle common numerical operations on 3-vectors of doubles.  
 * Unless otherwise noted, the operations are component by component, e.g.,
 * \verbatim
    v1 * v2 = [v1(0) *v2(0), v1(1)*v2(1), v1(2)*v2(2)]
 * \endverbatim
 *
 * This is intended as a small component which can be inexpensively created
 * on the stack; the constructors and destructor do not make use of
 * dynamic allocation.  The dvm3_Vector default constructor does 
 * not initialize the memory.
 * 
 *  Note that dvm3_Vector inherits from a struct, dvm3_VPOD, which 
 *  encapulates an array of doubles.  The inheritence is protected; 
 *  this implementation is exposed purely for efficiency considerations; 
 *  users shall not assume that this implementation detail will not 
 *  change in future.  DO NOT rely on the fact that there is an array 
 *  under the hood or on the naming of the data members within dvm3_VPOD.
 */
class dvm3_Vector
{
private:

  dvm3_VPOD data_;

  friend class dvm3_Matrix;
  friend class dvm3_RotMat;

public:

  // ---------------------------------------------------------------------
  /*!
   * \defgroup dtor_ctors_vector Destructor; Constructors
   */
  /*@{*/
  /**
   * Do-nothing destructor.
   */
 ~dvm3_Vector();

  /**
   * Default constructor; NO INITIALIZATION IS APPLIED.
   */
  dvm3_Vector();

  /**
   * Construct dvm3_Vector from 3 doubles.
   *
   * @param x   x component
   * @param y   y component
   * @param z   z component
   */
  dvm3_Vector( double x, double y, double z );

  /**
   * Copy constructor.
   *
   * @param other   vector to copy
   */
  dvm3_Vector( dvm3_Vector const& other );
    //
    //: copy constructor.

  /**
   * Copy operation: copy from a c-style vector.
   *
   * @param cv      vector to copy
   *
   *  REQUIREMENT:  cv has a length of at least 3 contiguous doubles
   *  and is appropriately aligned for doubles.
   */
  void copy_from_cvec( double const* cv );

  /**
   * Copy operation: copy to a c-style vector.
   *
   * @param v       vector copy
   *
   *  REQUIREMENT:  cv has a length of at least 3 contiguous doubles
   *  and is appropriately aligned for doubles.
   */
  void copy_to_cvec( double* v ) const;
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup itors Initializers
   */
  /*@{*/

  /**
   * Initialization: initialize with x, y, z.
   *
   * @param x   x component
   * @param y   y component
   * @param z   z component
   */
  void init( double x, double y, double z );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup assign Assignment, op= operators
   */
  /*@{*/
  /**
   * Copy-assignment operator
   *
   * @return       modified vector
   *
   * @param rhs    vector value for assignment
   */
  dvm3_Vector& operator=( dvm3_Vector const& rhs );

  /**
   * Assignment operator
   *
   * @return       modified vector
   *
   * @param rhs    value to be assigned
   */
  dvm3_Vector& operator=( double rhs );

  /**
   * Component-wise += operator.
   *
   * @return       modified vector
   *
   * @param rhs    value to be added
   */
  dvm3_Vector& operator+=( dvm3_Vector const& rhs );

  /**
   * Component-wise -= operator.
   *
   * @return       modified vector
   *
   * @param rhs    value to be subtracted
   */
  dvm3_Vector& operator-=( dvm3_Vector const& rhs );

  /**
   * Component-wise *= operator.
   *
   * @return       modified vector
   *
   * @param rhs    value to be multiplied
   */
  dvm3_Vector& operator*=( dvm3_Vector const& rhs );

  /**
   * Component-wise /= operator.
   *
   * @return       modified vector
   *
   * @param rhs    value to be divided by
   */
  dvm3_Vector& operator/=( dvm3_Vector const& rhs );

  /**
   * Add rhs to each component.
   *
   * @return       modified vector
   *
   * @param rhs    value to be added
   */
  dvm3_Vector& operator+=( double rhs );

  /**
   * Subtract rhs from each component.
   *
   * @return       modified vector
   *
   * @param rhs    value to be added
   */
  dvm3_Vector& operator-=( double rhs );

  /**
   * Multiply each component by rhs.
   *
   * @return       modified vector
   *
   * @param rhs    value
   */
  dvm3_Vector& operator*=( double rhs );

  /**
   * Divide each component by rhs.
   *
   * @return       modified vector
   *
   * @param rhs    value
   */
  dvm3_Vector& operator/=( double rhs );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup unary Unary operators
   */
  /*@{*/
  /**
   * Do-nothing unary +; provided for completeness.
   */
  dvm3_Vector& operator+();

  /**
   * Unary -; negates each component of this matrix
   */
  dvm3_Vector& operator-();
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup ators Accessors/Mutators
   */
  /*@{*/
  /**
   * Readonly access to component [i].
   * 
   * @return   const reference to component
   *
   * @param i  index
   */
  double const& operator[](int i) const;

  /**
   * Read/write access to component [i].
   *
   * @return   reference to component
   *
   * @param i  index
   */
  double& operator[](int i);

  /**
   * Copy of component [i].
   *
   * @return   copy of component [i]
   *
   * @param i  index
   */
  double  operator()(int i) const;

  /**
   * Read/write access to component [i].
   *
   * @return   reference to component
   *
   * @param i  index
   */
  double& operator()(int i);
    //
    //: read-write access to component [i].
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \ingroup norm
   */
  /*@{*/
  /**
   * normalize this dvm3_Vector; return original length vector length.
   *
   * @return  original length of this vector
   */
  double unitize();

  /**
   * normalize this dvm3_Vector; return original length vector length.
   *
   * @return  1 if vector is unit vector within tolerance tol.
   */
  int is_unit_vector( double const tol = 100.0 * DBL_EPSILON ) const;

  /**
   * Test whether this vector and the other vector are orthogonal within 
   * tolerance tol.
   *
   * @return 1 if this and other are orthogonal within tolerance tol.
   * 
   * @param tol  tolerance for determining orthogonality
   */

  int is_orthogonal_to( dvm3_Vector const& other,
                        double const tol = 100.0 * DBL_EPSILON ) const;
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup prods Dot, Vector products
   */
  /*@{*/
  /**
   * Dot product
   *
   * @return dot product (scalar product) of v with this dvm3_Vector.
   *
   * @param v   other vector
   */
  double dot( dvm3_Vector const& v ) const;

  /**
   * Cross product
   *
   * Set this dvm3_Vector to be the vector cross product of v1 and v2.
   * 
   * @param v1  1st vector
   * @param v2  2nd vector
   */
  void cross( dvm3_Vector const& v1, dvm3_Vector const& v2 );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup lincomb Linear combination
   */
  /*@{*/
  /**
   * Form linear combination:  this = c1 * v1 + c2 * v2.
   * \verbatim
       For each i:  v[i] = c1*v1[i] + c2*v[i]
   * \endverbatim
   *
   * @param c1    1st scalar
   * @param v1    1st vector
   * @param c2    2nd scalar
   * @param v2    2nd vector
   */
  void lincomb( double c1, dvm3_Vector const& v1,
                double c2, dvm3_Vector const& v2 );
  /*@}*/
  
  /*!
   * \ingroup norm
   */
  /*@{*/
  /** 
   * Normalize a vector
   *
   * @return original length of vi.
   *
   * @param vu   normalized version of vi
   * @param vi   vector
   */
  friend double unitize( dvm3_Vector& vu, dvm3_Vector const& vi );

  /*@}*/
  
  /*!
   * \ingroup prods
   */
  /*@{*/
  /** 
   * Dot product
   *
   * @return dot product of v1 and v2.
   *
   * @param v1   1st vector
   * @param v2   2nd vector
   */
  friend double dot( dvm3_Vector const& v1, dvm3_Vector const& v2 );

  /** 
   * Cross product
   *
   * @return cross product of v1 and v2.
   *
   * @param v1   1st vector
   * @param v2   2nd vector
   */
  friend void cross( dvm3_Vector& result, dvm3_Vector const& v1, 
                                          dvm3_Vector const& v2 );
    //
    //: result is vector (cross) product of v1 and v2.

  /*@}*/

  /*!
   * \ingroup lincomb
   */
  /*@{*/
  /** 
   * Linear combination c1 * v1 + c2 * v2.
   *
   * @param result linear combination c1 * v1 + c2 * v2.
   * @param c1     1st scalar
   * @param v1     1st vector
   * @param c2     2nd scalar
   * @param v2     2nd vector
   */
  friend void lincomb(            dvm3_Vector&       result, 
                       double c1, dvm3_Vector const& v1,
                       double c2, dvm3_Vector const& v2 );

  /*@}*/
  
  /*!
   * \defgroup mathops Componentwise math operations
   */
  /*@{*/
  /**
   * component-wise sum of v1 and v2.
   *
   * @result component-wise sum of v1 and v2
   *
   * @param v1  1st vector 
   * @param v2  2nd vector 
   */
  friend dvm3_Vector operator+ (dvm3_Vector const& v1, 
                                dvm3_Vector const& v2);

  /**
   * component-wise difference of v1 and v2.
   *
   * @result component-wise difference of v1 and v2
   *
   * @param v1  1st vector 
   * @param v2  2nd vector 
   */
  friend dvm3_Vector operator- (dvm3_Vector const& v1,
                                dvm3_Vector const& v2);

  /**
   * component-wise product of v1 and v2.
   *
   * @result component-wise product of v1 and v2
   *
   * @param v1  1st vector 
   * @param v2  2nd vector 
   */
  friend dvm3_Vector operator* (dvm3_Vector const& v1,
                                dvm3_Vector const& v2);

  friend dvm3_Vector operator/ (dvm3_Vector const& v1,
                                 dvm3_Vector const& v2);
    //
    //: component-wise division of v1 and v2.

  /**
   * component-wise sum: r + v[i]
   *
   * @result component-wise sum: r + v[i]
   *
   * @param r  scalar
   * @param v  vector 
   */
  friend dvm3_Vector operator+ (double       r, dvm3_Vector const& v);

  /**
   * component-wise difference: r - v[i]
   *
   * @result component-wise difference: r - v[i]
   *
   * @param r  scalar
   * @param v  vector 
   */
  friend dvm3_Vector operator- (double       r, dvm3_Vector const& v);

  /**
   * component-wise product: r * v[i]
   *
   * @result component-wise product: r * v[i]
   *
   * @param r  scalar
   * @param v  vector 
   */
  friend dvm3_Vector operator* (double       r, dvm3_Vector const& v);

  /**
   * component-wise division: r / v[i]
   *
   * @result component-wise division: r / v[i]
   *
   * @param r  scalar
   * @param v  vector 
   */
  friend dvm3_Vector operator/ (double       r, dvm3_Vector const& v);
    //
    //: set each component to r / component.

  /**
   * component-wise sum: v[i] + r
   *
   * @result component-wise sum: v[i] + r
   *
   * @param v  vector 
   * @param r  scalar
   */
  friend dvm3_Vector operator+ (dvm3_Vector const& v, double       r);

  /**
   * component-wise difference: v[i] - r
   *
   * @result component-wise difference: v[i] - r
   *
   * @param v  vector 
   * @param r  scalar
   */
  friend dvm3_Vector operator- (dvm3_Vector const& v, double       r);

  /**
   * component-wise product: v[i] * r
   *
   * @result component-wise product: v[i] * r
   *
   * @param v  vector 
   * @param r  scalar
   */
  friend dvm3_Vector operator* (dvm3_Vector const& v, double       r);

  /**
   * component-wise division: v[i] / r
   *
   * @result component-wise division: v[i] / r
   *
   * @param v  vector 
   * @param r  scalar
   */
  friend dvm3_Vector operator/ (dvm3_Vector const& v, double       r);
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup io I/O operations.
   */
  /*@{*/
  /**
   * Print this dvm3_Vector to an ostream.
   *
   * @param os       the ostream
   * @param prefix   optional prefix string 
   * @param postfix  optional postfix string
   */
  std::ostream& print_on( std::ostream& os, char const pre[] = "", 
                                            char const post[] = "" ) const;

  /**
   * Formatted dvm3_Matrix output to an ostream.
   *
   * @param os       the ostream
   */
  friend std::ostream& operator<<( std::ostream& os, dvm3_Vector const& );
    //
    //: formatted output.

  /**
   * Print this dvm3_Vector to a FILE* stream.
   *
   * @param of       the FILE*
   * @param v        vector to be printed
   * @param prefix   optional prefix string 
   * @param postfix  optional postfix string
   */
  void
  cprint_on( FILE* of, 
             char const prefix[] = "", char const postfix[] = "" );
  /*@}*/ 

  /*!
   * \ingroup prods
   */
  /*@{*/
  /**
   * Dyadic product
   */
  friend void 
  dyad_product(dvm3_Matrix&, const dvm3_Vector&, const dvm3_Vector&);

  /**
   * Multiply vector by matrix
   */
  friend void
  mvmult(dvm3_Vector&, const dvm3_Matrix&, const dvm3_Vector&);

  /**
   * Multiply vector by transpose of matrix
   */
  friend void
  mtvmult(dvm3_Vector&, const dvm3_Matrix&, const dvm3_Vector&);
};

//########################################################################
//########################################################################
//
//    #    #    #  #          #    #    #  ######   ####
//    #    ##   #  #          #    ##   #  #       #
//    #    # #  #  #          #    # #  #  #####    ####
//    #    #  # #  #          #    #  # #  #            #
//    #    #   ##  #          #    #   ##  #       #    #
//    #    #    #  ######     #    #    #  ######   ####
//
//########################################################################
//########################################################################

inline dvm3_Vector::
~dvm3_Vector()
{}

inline dvm3_Vector::
dvm3_Vector()
{}

inline dvm3_Vector::
dvm3_Vector( double x, double y, double z )
{
  vm_V3Math<double>::set( data_.x_, x, y, z );
}

inline dvm3_Vector::
dvm3_Vector( dvm3_Vector const& other )
  : data_( other.data_ )
{}

inline void dvm3_Vector::
init( double x, double y, double z )
{ vm_V3Math<double>::set( data_.x_, x, y, z ); }

inline dvm3_Vector& dvm3_Vector::
operator=( dvm3_Vector const& rhs )
{ vm_V3Math<double>::copy( data_.x_, rhs.data_.x_ );  return *this; }

inline dvm3_Vector& dvm3_Vector::
operator=( double rhs )
{ vm_V3Math<double>::set( data_.x_, rhs );   return *this; }

// This routine copies the source C-style vector to this DVector3
// using memcpy.  It is assumed that the source C vector points at storage
// containing at least 3 doubles

inline void dvm3_Vector::
copy_from_cvec( double const* vs )
{ vm_V3Math<double>::copy( data_.x_, vs ); }

// This routine copies the source C-style vector to the destination DVector3
// using memcpy.  It is assumed that the C vector points at storage
// containing at least 3 doubles

inline void dvm3_Vector::
copy_to_cvec( double* vd ) const
{ vm_V3Math<double>::copy( vd, data_.x_ ); }

#define _tpl_Op_EQ_(_OPNAM_,_OPEQ_) \
  inline dvm3_Vector& dvm3_Vector::operator _OPEQ_ ( dvm3_Vector const& v ) \
  { vm_V3Math<double>::_OPNAM_( data_.x_, v.data_.x_ );  return *this; } \
  inline dvm3_Vector& dvm3_Vector::operator _OPEQ_ ( double r ) \
  { vm_V3Math<double>::_OPNAM_( data_.x_, r );  return *this; }
  _tpl_Op_EQ_(add_eq,+=)
  _tpl_Op_EQ_(sub_eq,-=)
  _tpl_Op_EQ_(mul_eq,*=)
  _tpl_Op_EQ_(div_eq,/=)
#undef _tpl_Op_EQ_

inline dvm3_Vector& dvm3_Vector::operator+()
{ return *this; }

inline dvm3_Vector& dvm3_Vector::operator-()
{ vm_V3Math<double>::negate(data_.x_);  return *this; }

inline double const& dvm3_Vector::
operator[](int i) const
{ return data_.x_[i]; }

inline double& dvm3_Vector::
operator[](int i)
{ return data_.x_[i]; }

inline double dvm3_Vector::
operator()(int i) const
{ return data_.x_[i]; }

inline double& dvm3_Vector::
operator()(int i)
{ return data_.x_[i]; }

inline double dvm3_Vector::
dot( dvm3_Vector const& v ) const
{ return vm_V3Math<double>::dot( data_.x_, v.data_.x_ ); }

inline double
dot( dvm3_Vector const& v1, dvm3_Vector const& v2 )
{ return vm_V3Math<double>::dot( v1.data_.x_, v2.data_.x_ ); }

inline void dvm3_Vector::
cross( dvm3_Vector const& v1, dvm3_Vector const& v2 )
{ vm_V3Math<double>::cross( data_.x_, v1.data_.x_, v2.data_.x_ ); }

inline void
cross( dvm3_Vector& result, dvm3_Vector const& v1,
                             dvm3_Vector const& v2 )
{ vm_V3Math<double>::cross( result.data_.x_, v1.data_.x_, v2.data_.x_ ); }

inline double dvm3_Vector::
unitize()
{ return vm_V3Math<double>::unitize( data_.x_ ); }

inline double
unitize( dvm3_Vector       &vu,    // normalized version of vi
         dvm3_Vector const &vi     // input vector
)
{ return vm_V3Math<double>::unitize( vu.data_.x_, vi.data_.x_ ); }

inline int dvm3_Vector::
is_unit_vector( double const tol ) const
{ return vm_V3Math<double>::is_unit_vector( data_.x_, tol ); }

inline int dvm3_Vector::
is_orthogonal_to( dvm3_Vector const& other, double const tol ) const
{ return vm_V3Math<double>::are_orthogonal( data_.x_, other.data_.x_, tol ); }

inline void dvm3_Vector::
lincomb( double              c1,       // first scale factor
         dvm3_Vector const& v1,       // first vector
         double              c2,       // second scale factor
         dvm3_Vector const& v2        // second vector
       )
{ vm_V3Math<double>::lincomb( data_.x_, c1, v1.data_.x_, c2, v2.data_.x_ ); }

inline void
lincomb( dvm3_Vector& result, double c1, dvm3_Vector const& v1,
                               double c2, dvm3_Vector const& v2 )
{ 
  vm_V3Math<double>::lincomb( result.data_.x_, c1, 
                               v1.data_.x_, c2, v2.data_.x_ );
}

#define _tpl_Op_(_OP_) \
  inline dvm3_Vector operator _OP_ ( dvm3_Vector const& v1,  \
                                      dvm3_Vector const& v2 ) \
  { return dvm3_Vector( v1.data_.x_[0] _OP_ v2.data_.x_[0], \
                        v1.data_.x_[1] _OP_ v2.data_.x_[1], \
                        v1.data_.x_[2] _OP_ v2.data_.x_[2] ); } \
  inline dvm3_Vector operator _OP_ ( double r,  dvm3_Vector const& v2 ) \
  { return dvm3_Vector( r _OP_ v2.data_.x_[0], r _OP_ v2.data_.x_[1], \
                        r _OP_ v2.data_.x_[2] ); } \
  inline dvm3_Vector operator _OP_ ( dvm3_Vector const& v1, double r ) \
  { return dvm3_Vector( v1.data_.x_[0] _OP_ r, v1.data_.x_[1] _OP_ r, \
                        v1.data_.x_[2] _OP_ r ); }
  _tpl_Op_(+)
  _tpl_Op_(-)
  _tpl_Op_(*)
  _tpl_Op_(/)
#undef _tpl_Op_

inline std::ostream& dvm3_Vector::
print_on( std::ostream& os, char const prefix[], char const postfix[] ) const
{ return vm_V3Math<double>::print_on( os, data_.x_, prefix, postfix ); }

inline std::ostream&
operator<<( std::ostream& os, const dvm3_Vector& v )
{
  return v.print_on( os, "[", "]" );
}

inline void dvm3_Vector::
cprint_on( FILE* of, char const prefix[], char const postfix[] )
{ vm_V3Math<double>::cprint_on( of, data_.x_, prefix, postfix ); }

#endif
