# --8<--8<--8<--8<--
#
# Copyright (C) 2006 Smithsonian Astrophysical Observatory
#
# This file is part of testlib_sh
#
# src is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# src is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the 
#       Free Software Foundation, Inc. 
#       51 Franklin Street, Fifth Floor
#       Boston, MA  02110-1301, USA
#
# -->8-->8-->8-->8--

#################################################################

function test_initialize
{
  # name of the test executable
  _test_exe=${0##*/}

  _test_exitval=0

  _test_delete_logs=0
  _test_logfile_pattern=

  _test_toskip=0
  _test_todo=0
   test_todo=0

  _test_total=0
  _test_pass=0
  _test_fail=0
  _test_skip=0

  _test_todo_pass=0
  _test_todo_fail=0
  _test_todo_skip=0

  _test_result_hook=

  _test_num=1
  _test_subnum=0

  _test_label=

  # status of last test. todo is ignored.
  # 0 is ok, 1 is failure
  _test_status=

  _test_setup_vars

  # see if ndiff is available
  _test_have_ndiff=0
  test -x /opt/saotrace-2.0.5/bin//ndiff && _test_have_ndiff=1

  # should fiducial copies be made of files before comparison?
  _test_v_make_cmp_copy=0

  # should tests automatically test $? ?
  _test_v_check_exit_status=0

  # the variables that callers may modify. bash and ksh differ
  # on how to efficently set up arrays. do it the hard way
  _test_vars[${#_test_vars}]='make_cmp_copy'
  _test_vars[${#_test_vars}]='check_exit_status'

  # available hooks
  _test_hooks[${#_test_vars}]='result'


  echo "========================="
}


# set a testlib variable
function test_set
{
  typeset variable="$1"
  typeset    value="$2"

  typeset notfound=1

  for var in ${_test_vars[*]}; do
    if [ "$var" = "$variable" ]; then
      eval "_test_v_$variable='$value'"
      notfound=0
    fi
  done

  (( notfound )) &&  echo "# test_set: variable $variable does not exist"

  return $notfound
}

function _test_setup_vars
{
  _test_id=$(/usr/bin/printf '% 3d' $_test_num)
  test_num="$_test_num"
  test_id=$(/usr/bin/printf '%03d' $_test_num)

  _test_subid=$(/usr/bin/printf '%02d' $_test_subnum)
  test_subnum="$_test_subnum"
  test_subid=$(/usr/bin/printf '%02d' $_test_subnum)

  if (( _test_subnum )); then
    _test_label="$_test_id.$_test_subid"
  else
    _test_label="$_test_id"
  fi

  test_label="$_test_label"
}


# set up for next test
function _test_next
{
  (( test_status =  _test_status ))

  if (( _test_subnum )); then
    (( _test_subnum = _test_subnum + 1 ))
  else
    (( _test_num = _test_num + 1 ))
  fi 

  (( _test_total = _test_total + 1 ))
  _test_setup_vars
}


function test_begin_subtests
{
  if [ $_test_subnum -ne 0 ]; then
    echo "# error in test script: missing call to test_end_subtests"
    exit 1
  fi
  _test_subnum=1
  _test_setup_vars
}

function test_end_subtests
{
  if [ $_test_subnum -eq 0 ]; then
    echo "# error in test script: missing call to test_begin_subtests"
    exit 1
  fi
  _test_subnum=0

  # shadow the variables
  _test_setup_vars

  # call result_hook
  [[ $_test_last_fail != $_test_num ]]
  _test_result_hook $?

  # move on to next test
  (( _test_num = _test_num + 1 ))
  _test_setup_vars
}


#################################################################
# convenience function to set the shell status

function test_set_status { return $1; }


#################################################################
# set a testlib hook

function test_set_hook
{
  typeset     hook="$1"
  typeset function="$2"

  typeset notfound=1

  for var in ${_test_hooks[*]}; do
    if [ "$var" = "$hook" ]; then
      eval "_test_hook_$hook='$function'"
      notfound=0
    fi
  done

  (( notfound )) &&  echo "# test_set_hook: hook $hook does not exist"

  return $notfound
}

# set the result callback. for historical support only
function test_result_hook
{
  test_set_hook result $1
}

# invoke the results callback
function _test_result_hook
{
  if [ -n "$_test_hook_result" ]; then

    typeset status=$1

    # invert sense of status, as 0 is success
    (( status = ! status ))

    $_test_hook_result $status

  fi
}

function test_result_hook_save_output_on_failure {
  typeset pass=$1

  # if not in a subtest
  if (( test_subnum == 0 )) ; then

    # the test has succeeded, remove the test files
    if (( pass )); then
      /usr/bin/rm -f $$-${test_id}*

    # the test has failed:  save the test files by stripping the
    # $$- prefix
    else

      for src in $$-${test_id}*; do
        if [ -f "$src" ]; then
          dst="fail-${_test_exe}-${src#$$-}"
          mv "$src" "$dst"
          echo "# Failed results are in: $dst"
        fi
      done
    fi

  fi
}

#################################################################

function _test_init
{
  # save exit status of last command run
  _test_exit_status=$?

}

function _test_check_exit_status
{
  typeset result=0

  # if requested, check exit status
  # pass silently.  fail loudly.
  if (( _test_v_check_exit_status )); then 

    # fail if the status is ZERO
    if (( _test_v_check_exit_status == -1 )); then
      if (( _test_exit_status == 0 )) ; then
	_test_fail "$1: zero exit"
	result=1
      fi

    # fail if the status is NONZERO
    else
      if (( _test_exit_status )) ; then
	_test_fail "$1: non-zero exit"
	result=1
      fi
    fi
  fi

  return $result
}

function _test_finish
{
  if (( $_test_status )); then
    _test_last_fail=$_test_num
  fi

  _test_result_hook $_test_status
  _test_next
}

function _test_check_cmp_files
{
  typeset   new="$1"; shift
  typeset   old="$1"; shift
  typeset label="$1"; shift

  typeset error=0

  if [ ! -f "$new" ]; then

    _test_fail "$label"
    echo "# file '$new' doesn't exist"
    error=1
  fi

  if [ ! -f "$old" ]; then

    if (( _test_v_make_cmp_copy )); then
      cp "$new" "$old"
      if (( $? )); then
         (( error )) || _test_fail "$label"
	 echo "# make_cmp_copy: unable to copy '$new' to '$old'"
	 error=1
      else
         echo "# make_cmp_copy: copied '$new' to '$old'"
      fi

    else
      (( error )) || _test_fail "$label"
      echo "# file '$old' doesn't exist"
      error=1
    fi
  fi

  return $error
}


#################################################################

# status == 0 means success!
# ALL TESTS MUST (somehow) USE THIS, as it sets _test_status
function _test_it
{
  typeset  sense=$1;   shift
  typeset status=$1;   shift
  typeset  label="$1"; shift
  typeset result=0

  if [[ "$sense" == nok ]]; then
    (( status = ! status ))
  fi

  (( _test_status = status ))

  if (( _test_todo )); then

    if (( status )) ; then

      echo "Test $_test_label: not ok (todo): $label"
      (( _test_todo_fail = _test_todo_fail + 1 ))

    else

      echo "Test $_test_label:     ok: $label"
      (( _test_todo_pass = _test_todo_pass + 1 ))

    fi

  else

    if (( status )) ; then

      echo "Test $_test_label: not ok: $label"
      (( _test_fail = _test_fail + 1 ))
      result=1

    else

      echo "Test $_test_label:     ok: $label"
      (( _test_pass = _test_pass + 1 ))

    fi
  fi

  return $result
}


function test_ok
{
  _test_init
  _test_check_exit_status "$2" && _test_it ok "$@"
  _test_finish
}

function test_notok
{
  _test_init
  _test_check_exit_status "$2" && _test_it nok "$@"
  _test_finish
}

# synonym for test_notok
function test_nok
{
  test_notok "$@"
}

#################################################################

function _test_pass
{
  _test_it ok 0 "$1"
}

function _test_fail
{
  _test_it ok 1 "$1"
}

function test_pass
{
  _test_init
  _test_pass "$1"
  _test_finish
}

function test_fail
{
  _test_init
  _test_fail "$1"
  _test_finish
}


#################################################################

function test_skip
{
  typeset why="$1"
  typeset toskip="$2"

  echo "# Skip $toskip tests: $why"
  if (( _test_todo )); then
    (( _test_todo_skip = _test_todo_skip + $toskip ))
  else
    (( _test_skip = _test_skip + $toskip ))
  fi

  while (( toskip != 0 ))
  do
    (( toskip = toskip - 1 ))
    if (( _test_todo )); then
      echo "Test $_test_label:   skip (todo)"
    else
      echo "Test $_test_label:   skip"
    fi
    _test_next
  done
}

#################################################################

function test_begin_todo
{
  typeset what_todo="$1"
  _test_todo=1
  test_todo=1

  echo "# START todo: $what_todo"
}

function test_end_todo
{
  _test_todo=0
  test_todo=0
  echo "# END todo"

}


#################################################################
# numerical comparison functions using the shell

function _test_num
{
  typeset sense=$1;   shift
  typeset   msg="$1"; shift
  typeset   val=$1;   shift
  typeset   exp=$1;   shift
  typeset label="$1"; shift
  typeset status

  _test_check_exit_status "$label" || return

  (( status = val != exp ))

  _test_it $sense $status "$label"

  if (( _test_status )); then
    echo "#           got: $val"
    echo "# $msg: $exp"
  fi
}

function test_num_is
{
  _test_init
  _test_num ok  "     expected" "$@"
  _test_finish
}

function test_num_isnt
{
  _test_init
  _test_num nok "didn't expect" "$@"
  _test_finish
}

#################################################################
# numerical comparison functions using ndiff

function _test_ndiff
{
  typeset sense=$1;   shift
  typeset   msg="$1"; shift

  typeset ndiff_args

  OPTIND=
  while getopts :a: arg 
  do
   case $arg in
     a) ndiff_args="$OPTARG";;
     ?) shift $(( ${#*} -1 ))
	_test_fail "$1"
        echo "# illegal argument to test_ndiff_*: $OPTARG"
	return
	;;
   esac
  done

  shift $(($OPTIND -1 ))

  typeset   val=$1;   shift
  typeset   exp=$1;   shift
  typeset label="$1"; shift
  typeset status

  _test_check_exit_status "$label" || return

  # can't do nuthin if we ain't get ndiff
  if (( ! _test_have_ndiff )); then 
    _test_fail "$label"
    echo "# this test requires the ndiff program, which is not available"
    return
  fi

  echo "$val" > /tmp/$$-ndiff-val
  echo "$exp" | /opt/saotrace-2.0.5/bin//ndiff $ndiff_args - /tmp/$$-ndiff-val 1> /dev/null 2>&1

  _test_it $sense $? "$label"

  if (( _test_status )); then
    echo "#           got: $val"
    echo "# $msg: $exp"
  fi

  /usr/bin/rm -f /tmp/$$-ndiff-val

}


function test_ndiff
{
  _test_init
  _test_ndiff ok "     expected" "$@"
  _test_finish
}

function test_ndiff_is
{
  _test_init
  _test_ndiff ok "     expected" "$@"
  _test_finish
}

function test_ndiff_isnt
{
  _test_init
  _test_ndiff nok "didn't expect" "$@"
  _test_finish
}


#################################################################
# default string valued comparison
function _test_is
{
  typeset sense=$1;   shift
  typeset   msg="$1"; shift
  typeset   val=$1;   shift
  typeset   exp=$1;   shift
  typeset label="$1"; shift
  typeset status

  _test_check_exit_status "$label" || return

  [[ "$val" == "$exp" ]]
  status=$?
  _test_it $sense $status "$label"

  if (( _test_status )); then
    echo "#           got: $val"
    echo "# $msg: $exp"
  fi
}

function test_is
{
  _test_init
  _test_is ok  "     expected" "$@"
  _test_finish
}

function test_isnt
{
  _test_init
  _test_is nok "didn't expect" "$@"
  _test_finish
}

#################################################################
# compare files using cmp

function _test_file
{
  typeset sense="$1"; shift

  typeset fail

  OPTIND=
  while getopts :f: arg 
  do
   case $arg in
     f) fail="$OPTARG";;
     ?) shift $(( ${#*} -1 ))
	_test_fail "$1"
        echo "# illegal argument to test_file_*: $OPTARG"
	return
	;;
   esac
  done

  shift $(($OPTIND -1 ))

  typeset   new="$1"; shift
  typeset   old="$1"; shift
  typeset label="$1"; shift

  _test_check_exit_status "$label" || return

  test -z "$fail" && fail="$new"

  typeset status
  typeset error

  _test_check_cmp_files "$new" "$old" "$label"

  if (( ! $? )); then

    cmp "$old" "$new" > /dev/null 2>&1 
    _test_it $sense $? "$label"

    if (( _test_status )); then
      echo "#         got: $fail"
      echo "#    expected: $old"
    fi

  fi
}

function test_file
{
  _test_init
  _test_file ok "$@"
  _test_finish
}

function test_file_is
{
  _test_init
  _test_file ok "$@"
  _test_finish
}

function test_file_isnt
{
  _test_init
  _test_file nok "$@"
  _test_finish
}

#################################################################
# compare files using ndiff

function _test_numfile
{
  typeset sense="$1"; shift

  typeset ndiff_args
  typeset fail

  OPTIND=
  while getopts :a:f: arg 
  do
   case $arg in
     a) ndiff_args="$OPTARG";;
     f) fail="$OPTARG";;
     ?) shift $(( ${#*} -1 ))
	_test_fail "$1"
        echo "# illegal argument to test_file_*: $OPTARG"
	_test_finish
	return
	;;
   esac
  done

  shift $(($OPTIND -1 ))

  typeset   new="$1"; shift
  typeset   old="$1"; shift
  typeset label="$1"; shift

  _test_check_exit_status "$label" || return

  # can't do nuthin if we ain't get ndiff
  if (( ! _test_have_ndiff )); then 
    _test_fail "$label"
    echo "# this test requires the ndiff program, which is not available"
    return
  fi


  # failed file name defaults to new file
  test -z "$fail" && fail="$new"

  typeset status

  _test_check_cmp_files "$new" "$old" "$label"

  if (( ! $? )); then

    /opt/saotrace-2.0.5/bin//ndiff $ndiff_args "$old" "$new" > $$_test_ndiff.log 2> $$_test_ndiff.err
    typeset status=$?

    # ndiff doesn't output a correct status if one file is short
    # try this
    if [ 1 = $(wc -l < $$_test_ndiff.err) ] \
        && grep 'is short' $$_test_ndiff.err > /dev/null
    then 
      status=1
    fi

    _test_it $sense $status "$label"

    if (( _test_status )); then
      echo "#         got: $fail"
      echo "#    expected: $old"
    fi

    /usr/bin/rm -f $$_test_ndiff.log $$_test_ndiff.err

  fi
}

function test_numfile
{
  _test_init
  _test_numfile ok "$@"
  _test_finish
}

function test_numfile_is
{
  _test_init
  _test_numfile ok "$@"
  _test_finish
}

function test_numfile_isnt
{
  _test_init
  _test_numfile nok "$@"
  _test_finish
}

#################################################################
# grep for a string in a file

function _test_file_grep
{
  typeset sense=$1;  shift

  typeset grep_args
  typeset fail

  typeset grep=grep
  typeset matchfile=/dev/null

  OPTIND=
  while getopts :a:f:FEe: arg 
  do
   case $arg in
     a) grep_args="$OPTARG"  ;;

     f) matchfile="$OPTARG" ;;

     E) grep="/usr/bin/grep -E" ;;

     F) grep="/usr/bin/grep -F" ;;

     ?) shift $(( ${#*} -1 ))
	_test_fail "$1"
        echo "# illegal argument to test_file_grep_*: $OPTARG"
	return
	;;
   esac
  done

  shift $(($OPTIND -1 ))

  typeset pattern="$1"; shift
  typeset    file="$1"; shift
  typeset   label="$1"; shift

  _test_check_exit_status "$label" || return

  if [ -z "$pattern" ] ; then
    _test_fail "$label"
    echo "# test_file_grep: test setup error: empty grep pattern"
    return
  fi  

  typeset status

  if [ ! -f "$file" ]; then
    _test_fail "$label"
    echo "# file '$file' doesn't exist"
    return
  fi

  $grep  $grep_args "$pattern" $file > $matchfile 2> /dev/null
  _test_it $sense $? "$label"
}


function test_file_grep_is
{
  _test_init
  _test_file_grep ok "$@"
  _test_finish
}

function test_file_grep_isnt
{
  _test_init
  _test_file_grep nok "$@"
  _test_finish
}

#################################################################

function test_summary
{
  echo "-------------------------"
  echo "Total Tests: $_test_total"

  if (( _test_todo_pass )); then
    echo "  Passed: $_test_pass + $_test_todo_pass todo"
  else
    echo "  Passed: $_test_pass"
  fi

  if (( _test_todo_fail )); then
    echo "  Failed: $_test_fail + $_test_todo_fail todo"
  else
    echo "  Failed: $_test_fail"
  fi

  

  if (( _test_todo_skip )); then
    echo " Skipped: $_test_skip + $_test_todo_skip todo"
  else
    echo " Skipped: $_test_skip"
  fi

  echo "========================="
}


function test_exit
{
  [ $_test_fail -eq 0 ]
  exit $?
}
