#ifndef MATVEC_MACROS_H
#define MATVEC_MACROS_H


/** \file macros.h
 * \brief Macros dealing with transformations.
 *
 * These macros provide operations on matrices, transformation
 * matrices, as well as vector transformations (rotations and translations).
 */

#include <math.h>


/** \cond */

#define MATVEC_BEGIN do{
#define MATVEC_END } while(0)

/** \endcond */


/* =================================================================== */
/**
  \anchor extractors
  \name Matrix and Vector Element Extractors

  These macros extract elements from matrices and vectors.  They
  should be used wherever a macro parameter specifies an "extractor".
*/

/** @{ */

/** \cond */

/**
 * \internal
 * Generate a list of matrix elements for a 3X3 matrix.
 * This is a glue routine for \c MATVEC_MAT3X3_COMP_EVAL
 *
 * \hideinitializer
 *
 * @param matrix the variable containing the matrix
 * @param list   a list of matrix element offsets, usually given by
 *               the \c MATVEC_MAT3X3_AS_MDARRAY_COMP or
 *               \c MATVEC_MAT3X3_AS_ARRAY_COMP macros
 */
#define MATVEC_MAT3X3_COMP(matrix,list) \
        MATVEC_MAT3X3_COMP_EVAL(matrix,list)

/**
 * \internal
 * Generate a list of matrix elements. Usually invoked by
 * \c MATVEC_MAT3X3_COMP
 *
 * \hideinitializer
 *
 * @param matrix  the variable containing the matrix
 * @param m00     matrix[0][0]
 * @param m01     matrix[0][1]
 * @param m02     matrix[0][2]
 * @param m10     matrix[1][0]
 * @param m11     matrix[1][1]
 * @param m12     matrix[1][2]
 * @param m20     matrix[2][0]
 * @param m21     matrix[2][1]
 * @param m22     matrix[2][2]

 */
#define MATVEC_MAT3X3_COMP_EVAL( matrix,	\
				 m00, m01, m02,	\
				 m10, m11, m12,	\
				 m20, m21, m22)	\
    (matrix)m00,(matrix)m01,(matrix)m02,	\
    (matrix)m10,(matrix)m11,(matrix)m12,	\
    (matrix)m20,(matrix)m21,(matrix)m22


/**
 * \internal
 * Generate a list of matrix elements for a 4X4 matrix.
 * This is a glue routine for \c MATVEC_MAT4X4_COMP_EVAL
 *
 * \hideinitializer
 *
 * @param matrix the variable containing the matrix
 * @param list   a list of matrix element offsets, usually given by
 *               the \c MATVEC_MAT4X4_AS_MDARRAY_COMP or
 *               \c MATVEC_MAT4X4_AS_ARRAY_COMP macros
 */
#define MATVEC_MAT4X4_COMP(matrix,list) \
        MATVEC_MAT4X4_COMP_EVAL(matrix,list)

/**
 * \internal
 * Generate a list of matrix elements. Usually invoked by \c
 * MATVEC_MAT4X4_COMP
 *
 * \hideinitializer
 *
 * @param matrix  the variable containing the matrix
 * @param m00     matrix[0][0]
 * @param m01     matrix[0][1]
 * @param m02     matrix[0][2]
 * @param m02     matrix[0][3]
 * @param m10     matrix[1][0]
 * @param m11     matrix[1][1]
 * @param m12     matrix[1][2]
 * @param m13     matrix[1][3]
 * @param m20     matrix[2][0]
 * @param m21     matrix[2][1]
 * @param m22     matrix[2][2]
 * @param m23     matrix[2][3]
 * @param m30     matrix[2][0]
 * @param m31     matrix[2][1]
 * @param m32     matrix[2][2]
 * @param m33     matrix[2][3]

 */
#define MATVEC_MAT4X4_COMP_EVAL(		\
  matrix,					\
  m00, m01, m02, m03,				\
  m10, m11, m12, m13,				\
  m20, m21, m22, m23,				\
  m30, m31, m32, m33				\
  )						\
    (matrix)m00,(matrix)m01,(matrix)m02,(matrix)m03,	\
    (matrix)m10,(matrix)m11,(matrix)m12,(matrix)m13,	\
    (matrix)m20,(matrix)m21,(matrix)m22,(matrix)m23,	\
    (matrix)m30,(matrix)m31,(matrix)m32,(matrix)m33

/** \endcond */


/* ************************************************************ */
/* ************************************************************ */
/* Macros which return indices into arrays			*/


/** \cond */

/**
 * \internal
 * return indices into a 3x3 multi-dimensional matrix as a list
 *
 * \hideinitializer
 */
#define MATVEC_MAT3X3_AS_MDARRAY_COMP \
  [0][0],[0][1],[0][2],	\
  [1][0],[1][1],[1][2], \
  [2][0],[2][1],[2][2]

/**
 * \internal
 * return indices into a 4x4 multi-dimensional matrix as a list
 *
 * \hideinitializer
 */
#define MATVEC_MAT4X4_AS_MDARRAY_COMP \
    [0][0],[0][1],[0][2],[0][3],	\
    [1][0],[1][1],[1][2],[1][3],	\
    [2][0],[2][1],[2][2],[2][3],	\
    [3][0],[3][1],[3][2],[3][3]

/**
 * \internal
 * return indices for the upper left 3x3 matrix from a "flat" 2D array
 * as a list
 * \hideinitializer
 *
 * \param ncols		the number of columns in the matrix
*/
#define MATVEC_MAT3X3_AS_ARRAY_COMP(ncols)	\
    [0*(ncols)+0],[0*(ncols)+1],[0*(ncols)+2],	\
    [1*(ncols)+0],[1*(ncols)+1],[1*(ncols)+2],	\
    [2*(ncols)+0],[2*(ncols)+1],[2*(ncols)+2]


/**
 * \internal
 * return indices for the upper left 4x4 (sub)matrix from a "flat" 2D array
 * as a list
 * \hideinitializer
 *
 * \param ncols		the number of columns in the matrix
*/
#define MATVEC_MAT4X4_AS_ARRAY_COMP(ncols)			\
    [0*(ncols)+0],[0*(ncols)+1],[0*(ncols)+2],[0*(ncols)+3],	\
    [1*(ncols)+0],[1*(ncols)+1],[1*(ncols)+2],[1*(ncols)+3],	\
    [2*(ncols)+0],[2*(ncols)+1],[2*(ncols)+2],[2*(ncols)+3],	\
    [3*(ncols)+0],[3*(ncols)+1],[3*(ncols)+2],[3*(ncols)+3]


/** \endcond */


/* ************************************************************ */
/* ************************************************************ */
/* Macros which return elements of arrays			*/


/* ************************************************************ */
/* Multi-dimensional array storage				*/

/**
 * \brief Extract the elements of the upper left 3x3 (sub)matrix from
 * a two-dimensional C array.
 *
 * \hideinitializer
 *
 * @param matrix the variable containing the matrix
 * 
 * \c matrix should resolve into an array specification which can be
 * indexed using the standard C bracket operators.  For example:
 * \code
 double matrix[3][3];
 MATVEC_MAT3X3_AS_MDARRAY(matrix)

 void func( double matrix[][3] )
 {
    MATVEC_MAT3X3_AS_MDARRAY(matrix)
 }

 * \endcode
 */
#define MATVEC_MAT3X3_AS_MDARRAY(matrix) \
  MATVEC_MAT3X3_COMP(matrix,MATVEC_MAT3X3_AS_MDARRAY_COMP)


/**
 * \brief Extract the elements of the upper left 4x4 (sub)matrix from
 * a two-dimensional C array.
 *
 * \hideinitializer
 *
 * @param matrix the variable containing the matrix
 * 
 * \c matrix should resolve into an array specification which can be
 * indexed using the standard C bracket operators.  For example:
 * \code
 double matrix[4][4];
 MATVEC_MAT4X4_AS_MDARRAY(matrix)

 void func( double matrix[][4] )
 {
    MATVEC_MAT4X4_AS_MDARRAY(matrix)
 }

 * \endcode
 */
#define MATVEC_MAT4X4_AS_MDARRAY(matrix) \
  MATVEC_MAT4X4_COMP(matrix,MATVEC_MAT4X4_AS_MDARRAY_COMP)


/* ************************************************************ */
/* "Flat" array storage						*/

/**
 * \brief Extract elements for the upper left 3x3 (sub)matrix from a matrix
        represented as a flat C array.
 *  
 * \hideinitializer
 *
 * \param	matrix the flat (one-dimensional) array containing the matrix
 * \param	ncols the number of columns in the matrix. this parameter is
 *		used multiple times, so should not cause any side effects

 * The macro extracts the upper left 3x3 matrix from a two dimensional
 * matrix represented by a flat (one-dimensional array).  The input
 * matrix may be larger than 3x3.

 * \c matrix should resolve into an array specification which can be
 * indexed using the standard C bracket operators.  For example:

 * \code
 double matrix[3*3];
 MATVEC_MAT3X3_AS_ARRAY(matrix,3)

 double matrix[4*4];
 MATVEC_MAT3X3_AS_ARRAY(matrix,4)

 void func( double *matrixa, double matrixb[] )
 {
    MATVEC_MAT3X3_AS_ARRAY(matrixa,3)
    MATVEC_MAT3X3_AS_ARRAY(matrixb,3)
 }

 * \endcode
 *
 */
#define MATVEC_MAT3X3_AS_ARRAY(matrix,ncols)			\
  MATVEC_MAT3X3_COMP(matrix,MATVEC_MAT3X3_AS_ARRAY_COMP(ncols))



/**
 * \brief Extract elements for the upper left 4x4 matrix from a matrix
        represented as a flat C array.
 *  
 * \hideinitializer
 *
 * \param	matrix the flat (one-dimensional) array containing the matrix
 * \param	ncols the number of columns in the matrix. this parameter is
 *		used multiple times, so should not cause any side effects

 * The macro extracts the upper left 4x4 matrix from a two dimensional
 * matrix represented by a flat (one-dimensional array).  The input
 * matrix may be larger than 4x4.

 * \c matrix should resolve into an array specification which can be
 * indexed using the standard C bracket operators.  For example:

 * \code
 double matrixa[4*4];
 MATVEC_MAT4X4_AS_ARRAY(matrixa,4)

 double matrixb[5*5];
 MATVEC_MAT4X4_AS_ARRAY(matrixb,5)

 void func( double *matrixa, double matrixb[] )
 {
    MATVEC_MAT4X4_AS_ARRAY(matrixa,4)
    MATVEC_MAT4X4_AS_ARRAY(matrixb,5)
 }

 * \endcode
 *
 */
#define MATVEC_MAT4X4_AS_ARRAY(matrix,ncols)			\
  MATVEC_MAT4X4_COMP(matrix,MATVEC_MAT4X4_AS_ARRAY_COMP(ncols))


/* ************************************************************ */
/* ************************************************************ */
/* Extract vector components					*/


/**
 * \brief Extract components for a 3 component vector from an array
 * 
 * \hideinitializer
 *
 * @param vector a pointer to the first element in the vector
 * @param stride the number of array elements between vector
 *               elements. for typical uses this is 1.
 */
#define MATVEC_VEC_AS_ARRAY(vector,stride) \
  (vector)[0],(vector)[stride],(vector)[2*stride]

/**
 * \brief Extract components for a 3 component vector from a structure.
 *
 * \hideinitializer
 *
 * @param vector  the structure containing the vector.  this is \b not
 *                a pointer to the structure. for pointers, pass \c *ptr
 * @param x       the name of the x component structure element
 * @param y       the name of the y component structure element
 * @param z       the name of the z component structure element

 For example,

 \code
 struct { double x, y, z } vector, *vecptr;
 MATVEC_VEC_AS_STRUCT(vector,x,y,z)
 MATVEC_VEC_AS_STRUCT(*vecptr,x,y,z)
 \endcode

 */
#define MATVEC_VEC_AS_STRUCT(vector,x,y,z)	\
  (vector).x,(vector).y,(vector).z

/** @} */





/*===========================================================================*/

/** \name Basic Matrix Operations
 *
 */

/** \{ */

/** \cond */
/** 
 * \internal
 * Assign 3X3 values to 3X3 destinations.  Values and destinations
 * can be anything.  See  ::MATVEC_MAT3X3_ASSIGN for an example
 *
 * \hideinitializer
 *
 * @param d00..d22	3X3 destination elements
 * @param s00..s22	3X3 source elements
 *
 */
#define MATVEC_MAT3X3_ASSIGN_COMP(	\
  d00, d01, d02,			\
  d10, d11, d12,			\
  d20, d21, d22,			\
  s00, s01, s02,			\
  s10, s11, s12,			\
  s20, s21, s22				\
  )					\
  MATVEC_BEGIN				\
  (d00) = (s00);			\
  (d01) = (s01);			\
  (d02) = (s02);			\
  (d10) = (s10);			\
  (d11) = (s11);			\
  (d12) = (s12);			\
  (d20) = (s20);			\
  (d21) = (s21);			\
  (d22) = (s22);			\
  MATVEC_END

/** 
 * \internal
 * Assign 4X4 values to 4X4 destinations.  Values and destinations
 * can be anything.  See  ::MATVEC_MAT4X4_ASSIGN for an example
 *
 * \hideinitializer
 *
 * @param d00..d33	4x4 destination elements
 * @param s00..s33	4x4 source elements
 *
 */
#define MATVEC_MAT4X4_ASSIGN_COMP(	\
  d00, d01, d02, d03,			\
  d10, d11, d12, d13,			\
  d20, d21, d22, d23,			\
  d30, d31, d32, d33,			\
  s00, s01, s02, s03,			\
  s10, s11, s12, s13,			\
  s20, s21, s22, s23,			\
  s30, s31, s32, s33			\
  )					\
  MATVEC_BEGIN				\
  (d00) = (s00);			\
  (d01) = (s01);			\
  (d02) = (s02);			\
  (d03) = (s03);			\
  (d10) = (s10);			\
  (d11) = (s11);			\
  (d12) = (s12);			\
  (d13) = (s13);			\
  (d20) = (s20);			\
  (d21) = (s21);			\
  (d22) = (s22);			\
  (d23) = (s23);			\
  (d30) = (s30);			\
  (d31) = (s31);			\
  (d32) = (s32);			\
  (d33) = (s33);			\
  MATVEC_END

/** \endcond */

/**
 * Assign a 3X3 upper-left (sub)matrix 
 * \hideinitializer
 *
 * \param dst	the destination matrix extractor
 * \param src	the source matrix extractor
 *
 * Assign a 3X3 matrix to another
 *
 * Example:
 \code
 double mata[3][3];
 double *matb = malloc((3*3) * sizeof(double));

 MATVEC_MAT3X3_ASSIGN(
 	MATVEC_MAT3X3_AS_ARRAY(matb),
 	MATVEC_MAT3X3_AS_MDARRAY(mata)
	);
 \endcode
 */
#define MATVEC_MAT3X3_ASSIGN(dst,src)	\
  MATVEC_MAT3X3_ASSIGN_COMP(dst,src)


/**
 * Assign a 4X4 upper-left (sub)matrix 
 * \hideinitializer
 *
 * \param dst	the destination matrix extractor
 * \param src	the source matrix extractor
 *
 *
 * Example:
 \code
 double mata[4][4];
 double *matb = malloc((4*4) * sizeof(double));

 MATVEC_MAT4X4_COPY(
 	MATVEC_MAT4X4_AS_ARRAY(matb),
 	MATVEC_MAT4X4_AS_MDARRAY(mata) );
 \endcode
 */
#define MATVEC_MAT4X4_ASSIGN(dst,src)	\
  MATVEC_MAT4X4_ASSIGN_COMP(dst,src)



/** 
 * Set a 3X3 upper-left (sub)matrix to a given value.
 *
 * \hideinitializer
 *
 * \param matrix	the (sub)matrix extractor
 * \param value		the value to which the matrix entries are to be
 *	 		set. this is used multiple times, so should not have
 *			any side-effects.
 */
#define MATVEC_MAT3X3_SET(matrix,value)	\
  MATVEC_MAT3X3_ASSIGN_COMP(		\
    matrix,				\
    (value),(value),(value),		\
    (value),(value),(value),		\
    (value),(value),(value)		\
    )


/** 
 * Set a 4X4 upper-left (sub)matrix to a given value.
 *
 * \hideinitializer
 *
 * \param matrix	the (sub)matrix extractor
 * \param value		the value to which the matrix entries are to be
 *	 		set. this is used multiple times, so should not have
 *			any side-effects.
 */
#define MATVEC_MAT4X4_SET(matrix,value)	\
  MATVEC_MAT4X4_ASSIGN_COMP(		\
    matrix,				\
    (value),(value),(value),(value),	\
    (value),(value),(value),(value),	\
    (value),(value),(value),(value),	\
    (value),(value),(value),(value),	\
    )


/** 
 * Set a 3X3 upper-left (sub)matrix to the identity matrix
 *
 * \hideinitializer
 *
 * \param matrix	the (sub)matrix extractor
 */
#define MATVEC_MAT3X3_IDENTITY(matrix)	\
    MATVEC_MAT3X3_ASSIGN_COMP(		\
      matrix,				\
      1,0,0,				\
      0,1,0,				\
      0,0,1				\
      )


/** 
 * Set a 4X4 upper-left (sub)matrix to the identity matrix
 *
 * \hideinitializer
 *
 * \param matrix	the (sub)matrix extractor
 */
#define MATVEC_MAT4X4_IDENTITY(matrix)	\
  MATVEC_MAT4X4_ASSIGN_COMP(			\
    matrix,					\
    1,0,0,0,					\
    0,1,0,0,					\
    0,0,1,0,					\
    0,0,0,1					\
    )



/** \cond */
/** 
 * \internal
 * Transpose 3X3 values to 3X3 destinations.  Values and destinations
 * can be anything.  See  ::MATVEC_MAT3X3_TRANSPOSE for an example
 *
 * \hideinitializer
 *
 * @param d00..d22	3X3 result elements
 * @param s00..s22	3X3 input elements
 *
 */
#define MATVEC_MAT3X3_TRANSPOSE_COMP(	\
  d00, d01, d02,			\
  d10, d11, d12,			\
  d20, d21, d22,			\
  s00, s01, s02,			\
  s10, s11, s12,			\
  s20, s21, s22				\
  )					\
  MATVEC_BEGIN				\
  (d00) = (s00);			\
  (d01) = (s10);			\
  (d02) = (s20);			\
  (d10) = (s01);			\
  (d11) = (s11);			\
  (d12) = (s21);			\
  (d20) = (s02);			\
  (d21) = (s12);			\
  (d22) = (s22);			\
  MATVEC_END

/** 
 * \internal
 * Transpose 4X4 values to 4X4 destinations.  Values and destinations
 * can be anything.  See  ::MATVEC_MAT4X4_TRANSPOSE for an example
 *
 * \hideinitializer
 *
 * @param d00..d33	4x4 result elements
 * @param s00..s33	4x4 input elements
 *
 */
#define MATVEC_MAT4X4_TRANSPOSE_COMP(	\
  d00, d01, d02, d03,			\
  d10, d11, d12, d13,			\
  d20, d21, d22, d23,			\
  d30, d31, d32, d33,			\
  s00, s01, s02, s03,			\
  s10, s11, s12, s13,			\
  s20, s21, s22, s23,			\
  s30, s31, s32, s33			\
  )					\
  MATVEC_BEGIN				\
  (d00) = (s00);			\
  (d01) = (s10);			\
  (d02) = (s20);			\
  (d03) = (s30);			\
  (d10) = (s01);			\
  (d11) = (s11);			\
  (d12) = (s21);			\
  (d13) = (s31);			\
  (d20) = (s02);			\
  (d21) = (s12);			\
  (d22) = (s22);			\
  (d23) = (s32);			\
  (d30) = (s03);			\
  (d31) = (s13);			\
  (d32) = (s23);			\
  (d33) = (s33);			\
  MATVEC_END

/** \endcond */

/** 
 * Transpose a 3X3 upper-left (sub)matrix
 *
 * \hideinitializer
 *
 * \param res	the result (sub)matrix extractor.
 * \param src	the input (sub)matrix extractor
 *
 * Transpose a 3X3 upper-left (sub)matrix, storing the results in
 * another matrix.
 */
#define MATVEC_MAT3X3_TRANSPOSE(res,src)	\
  MATVEC_MAT3X3_TRANSPOSE_COMP(res,src)

/** 
 * Transpose a 4X4 upper-left (sub)matrix
 *
 * \hideinitializer
 *
 * \param res	the result (sub)matrix extractor
 * \param dst	the output (sub)matrix extractor.
 *
 * Transpose a 4X4 upper-left (sub)matrix, storing the results in
 * another matrix.
 */
#define MATVEC_MAT4X4_TRANSPOSE(res,src)	\
  MATVEC_MAT4X4_TRANSPOSE_COMP(res,src)


/** \cond */
/** 
 * \internal
 * Multiply 3X3 matrices  Values and destinations
 * can be anything.  See  ::MATVEC_MAT3X3_MULT for an example
 *
 * \hideinitializer
 *
 * @param d00..d22	3X3 result elements
 * @param a00..a22	3X3 matrix a elements
 * @param b00..b22	3X3 matrix b elements
 *
 */
#define MATVEC_MAT3X3_MULT_COMP(				\
  d00, d01, d02,						\
  d10, d11, d12,						\
  d20, d21, d22,						\
  a00, a01, a02,						\
  a10, a11, a12,						\
  a20, a21, a22,						\
  b00, b01, b02,						\
  b10, b11, b12,						\
  b20, b21, b22							\
  )								\
  MATVEC_BEGIN							\
  (d00) = (a00) * (b00) + (a01) * (b10) + (a02) * (b20);	\
  (d01) = (a00) * (b01) + (a01) * (b11) + (a02) * (b21);	\
  (d02) = (a00) * (b02) + (a01) * (b12) + (a02) * (b22);	\
  (d10) = (a10) * (b00) + (a11) * (b10) + (a12) * (b20);	\
  (d11) = (a10) * (b01) + (a11) * (b11) + (a12) * (b21);	\
  (d12) = (a10) * (b02) + (a11) * (b12) + (a12) * (b22);	\
  (d20) = (a20) * (b00) + (a21) * (b10) + (a22) * (b20);	\
  (d21) = (a20) * (b01) + (a21) * (b11) + (a22) * (b21);	\
  (d22) = (a20) * (b02) + (a21) * (b12) + (a22) * (b22);	\
  MATVEC_END



/**
 * \internal
 * Multiply 4X4 matrices  Values and destinations
 * can be anything.  See  ::MATVEC_MAT4X4_MULT for an example
 *
 * \hideinitializer
 *
 * @param d00..d33      4X4 result elements
 * @param a00..a33      4X4 matrix a elements
 * @param b00..b33      4X4 matrix b elements
 *
 */
#define MATVEC_MAT4X4_MULT_COMP(					\
  d00, d01, d02, d03,							\
  d10, d11, d12, d13,							\
  d20, d21, d22, d23,							\
  d30, d31, d32, d33,							\
  a00, a01, a02, a03,							\
  a10, a11, a12, a13,							\
  a20, a21, a22, a23,							\
  a30, a31, a32, a33,							\
  b00, b01, b02, b03,							\
  b10, b11, b12, b13,							\
  b20, b21, b22, b23,							\
  b30, b31, b32, b33							\
  )									\
  MATVEC_BEGIN								\
  (d00) = (a00) * (b00) + (a01) * (b10) + (a02) * (b20) + (a03) * (b30); \
  (d01) = (a00) * (b01) + (a01) * (b11) + (a02) * (b21) + (a03) * (b31); \
  (d02) = (a00) * (b02) + (a01) * (b12) + (a02) * (b22) + (a03) * (b32); \
  (d03) = (a00) * (b03) + (a01) * (b13) + (a02) * (b23) + (a03) * (b33); \
  (d10) = (a10) * (b00) + (a11) * (b10) + (a12) * (b20) + (a13) * (b30); \
  (d11) = (a10) * (b01) + (a11) * (b11) + (a12) * (b21) + (a13) * (b31); \
  (d12) = (a10) * (b02) + (a11) * (b12) + (a12) * (b22) + (a13) * (b32); \
  (d13) = (a10) * (b03) + (a11) * (b13) + (a12) * (b23) + (a13) * (b33); \
  (d20) = (a20) * (b00) + (a21) * (b10) + (a22) * (b20) + (a23) * (b30); \
  (d21) = (a20) * (b01) + (a21) * (b11) + (a22) * (b21) + (a23) * (b31); \
  (d22) = (a20) * (b02) + (a21) * (b12) + (a22) * (b22) + (a23) * (b32); \
  (d23) = (a20) * (b03) + (a21) * (b13) + (a22) * (b23) + (a23) * (b33); \
  (d30) = (a30) * (b00) + (a31) * (b10) + (a32) * (b20) + (a33) * (b30); \
  (d31) = (a30) * (b01) + (a31) * (b11) + (a32) * (b21) + (a33) * (b31); \
  (d32) = (a30) * (b02) + (a31) * (b12) + (a32) * (b22) + (a33) * (b32); \
  (d33) = (a30) * (b03) + (a31) * (b13) + (a32) * (b23) + (a33) * (b33); \
  MATVEC_END


/** \endcond */

/**
 * Mutiply two 3X3 Upper-left (sub)matrices
 *
 * \hideinitializer
 *
 * \param res	the result (sub)matrix extractor.
 * \param a	the first (sub)matrix extractor
 * \param b	the second (sub)matrix extractor.
 *
 * Multiply two 3X3 upper-left (sub)matrices, storing the result in
 * another matrix.
 */
#define MATVEC_MAT3X3_MULT(res,a,b)	\
  MATVEC_MAT3X3_MULT_COMP(res,a,b)

/**
 * Mutiply two 4X4 Upper-left (sub)matrices
 *
 * \hideinitializer
 *
 * \param res	the output (sub)matrix extractor.
 * \param a	the first (sub)matrix extractor
 * \param b	the second (sub)matrix extractor.
 *
 * Multiply two 4X4 upper-left (sub)matrices, storing the result in
 * another matrix.
 */
#define MATVEC_MAT4X4_MULT(res,a,b)	\
  MATVEC_MAT4X4_MULT_COMP(res,a,b)



/** \} */

/*===========================================================================*/

/** \name Transformation Matrix Operations
 *
 */

/** \{ */

/** 
 * Set a 3X3 matrix to a rotation matrix representing rotation of the
 * <em>coordinate frame</em> about the X axis
 *
 * \hideinitializer
 * \param matrix	result 3x3 (sub)matrix extractor
 * \param theta		rotation angle in radians

 * For positive \f$\theta\f$, a rotation about X carries the Y axis
 * toward the Z axis.
 *
 *  \f{eqnarray*} {\rm rotX}(\theta) &=& \left[ 
 *                  \begin{array}{ccc}
 *                      1          &  0          &  0          \\
 *                      0          &  \cos\theta &  \sin\theta \\
 *                      0          & -\sin\theta &  \cos\theta
 *                  \end{array}
 *                \right]
 * \f}
 */
#define MATVEC_MAT3X3_ROT_X(matrix,theta)	\
  MATVEC_BEGIN					\
    double cos_theta = cos(theta);		\
    double sin_theta = sin(theta);		\
    MATVEC_MAT3X3_ASSIGN_COMP(			\
      matrix,					\
      1,     0,         0,			\
      0,  cos_theta, sin_theta,			\
      0, -sin_theta, cos_theta			\
      );					\
  MATVEC_END

/** 
 * Set a 3X3 matrix to a rotation matrix representing rotation of the
 * <em>coordinate frame</em> about the Y axis
 *
 * \hideinitializer
 * \param matrix	result 3x3 (sub)matrix extractor
 * \param theta		rotation angle in radians
 *
 * For positive \f$\theta\f$, a rotation about Y carries the Z axis
 * toward the X axis.
 *
 *  \f{eqnarray*} {\rm rotY}(\theta) &=& \left[ 
 *                  \begin{array}{ccc}
 *                      \cos\theta &  0          & -\sin\theta \\
 *                      0          &  1          &  0          \\
 *                      \sin\theta &  0          &  \cos\theta
 *                  \end{array}
 *                \right]
 * \f}
 */
#define MATVEC_MAT3X3_ROT_Y(matrix,theta)	\
  MATVEC_BEGIN					\
    double cos_theta = cos(theta);		\
    double sin_theta = sin(theta);		\
    MATVEC_MAT3X3_ASSIGN_COMP(			\
      matrix,					\
      cos_theta, 0,   -sin_theta,		\
      0,         1,       0,			\
      sin_theta, 0,    cos_theta		\
      );					\
  MATVEC_END

/** 
 * Set a 3X3 matrix to a rotation matrix representing rotation of the
 * <em>coordinate frame</em> about the Z axis
 *
 * \hideinitializer
 * \param matrix	result 3x3 (sub)matrix extractor
 * \param theta		rotation angle in radians
 *
 * For positive \f$\theta\f$, a rotation about Z carries the X axis
 * toward the Y axis.
 *
 *  \f{eqnarray*} {\rm rotZ}(\theta) &=& \left[ 
 *                  \begin{array}{ccc}
 *                      \cos\theta &  \sin\theta &  0          \\
 *                     -\sin\theta &  \cos\theta &  0          \\
 *                      0          &  0          &  1
 *                  \end{array}
 *                \right]
 * \f}
 */
#define MATVEC_MAT3X3_ROT_Z(matrix,theta)	\
  MATVEC_BEGIN					\
    double cos_theta = cos(theta);		\
    double sin_theta = sin(theta);		\
    MATVEC_MAT3X3_ASSIGN_COMP(			\
      matrix,					\
      cos_theta, sin_theta, 0,			\
      -sin_theta, cos_theta, 0,			\
      0,         0,     1			\
      );					\
  MATVEC_END



    
/**
 * Set a 3x3 (sub)matrix to a rotation matrix representing rotation of
 * the <em>coordinate system</em> through a modified YXZ Euler angle
 * rotation sequence (\c YMXZ stands for \c Y, <em>minus</em> X, \c Z).  
 *
 * This system can be thought of as representing an alt-az system with
 * the horizon in the X-Z plane.  The system differs from the standard
 * Euler YXZ system in that the sense of the rotation about the X axis
 * is reversed (Z rotates towarsd Y) to match the common definition of
 * altitude.
 *
 * \hideinitializer
 *
 * @param matrix    result 3x3 (sub)matrix extractor
 *
 * @param azimuth   The azimuth angle (in radians).
 *                  This is the rotation angle about the current Y axis.
 *                  Positive azimuth rotates Z towards X.
 *
 * @param elevation The elevation angle (in radians). This is the
 * 		    rotation angle about the new X axis.
 * 		    Positive elevation rotates Z towards Y, which is
 *                  opposite from the standard convention.
 *
 * @param clock     The clock angle (in radians). This is the rotation
 *                  angle about the new Z axis.  Positive clocking
 *                  rotates X towards Y.
 *
 * The new coordinate frame is generated via the equivalent of

 * \li rotating the input frame about its Y axis by \c azimuth
 * \li rotating the new frame about its X axis by \c -elevation
 * \li rotating the new frame about its Z axis by \c clock
 *
 * Let \f$\alpha\f$ be the azimuth angle, \f$\lambda\f$ be the 
 * elevation angle, and \f$\zeta\f$ the clocking angle about the Z axis.
 * Then, in detail
 *    \f${\bf A} = {\bf B} {\bf C} {\bf D}\f$ where
 *
 * \f{eqnarray*} {\bf D} &=& \left[ 
 *                 \begin{array}{ccc}
 *                     \cos\alpha & 0    & -\sin\alpha \\
 *                     0          & 1    &  0          \\
 *                     \sin\alpha & 0    &  \cos\alpha
 *                 \end{array}
 *               \right] \\
 *     {\bf C} &=& \left[ 
 *                 \begin{array}{ccc}
 *                     1    & 0           &  0           \\
 *                     0    & \cos\lambda & -\sin\lambda \\
 *                     0    & \sin\lambda &  \cos\lambda
 *                 \end{array}
 *               \right] \\
 *     {\bf B} &=& \left[ 
 *                 \begin{array}{ccc}
 *                     \cos\zeta &  \sin\zeta & 0 \\
 *                    -\sin\zeta &  \cos\zeta & 0 \\
 *                     0         & 0          & 1 
 *                 \end{array}
 *               \right]
 * \f}
 * so that
 * \f{eqnarray*} {\bf A} &=& 
 * \left[ 
 * \begin{array}{ccc}
 *   \cos\zeta \cos\alpha - \sin\zeta \sin\lambda \sin\alpha &
 *                          \sin\zeta \cos\lambda            &
 *  -\cos\zeta \sin\alpha - \sin\zeta \sin\lambda \cos\alpha \\

 *  -\sin\zeta \cos\alpha - \cos\zeta \sin\lambda \cos\alpha &
 *                          \cos\zeta \cos\lambda            &
 *   \sin\zeta \sin\alpha - \cos\zeta \sin\lambda \cos\alpha \\

 *   \cos\lambda \sin\alpha  &
 *   \sin\lambda             &
 *   \cos\lambda \cos\alpha  
 * \end{array}
 * \right]
 * \f}
 *
 * In terms of the MATVEC rotation macros, this would be
 * \f${\bf A = {\bf B} {\bf C} {\bf D}\f$ where:
 * \code
 *   MATVEC_MAT3X3_ROT_Y(D, azimuth);
 *   MATVEC_MAT3X3_ROT_X(C, -elevation);
 *   MATVEC_MAT3X3_ROT_Z(B, clocking);
 * \endcode
 *
 */

#define MATVEC_MAT3X3_ROT_YMXZ(				\
  matrix,azimuth,elevation,clock			\
  )							\
  MATVEC_BEGIN						\
    double cos_az = cos(azimuth);			\
    double sin_az = sin(azimuth);			\
    double cos_el = cos(elevation);			\
    double sin_el = sin(elevation);			\
    double cos_z  = cos(clock);				\
    double sin_z  = sin(clock);				\
    MATVEC_MAT3X3_ASSIGN_COMP(				\
      matrix,						\
      cos_z * cos_az - sin_z * sin_el * sin_az,		\
      sin_z * cos_el,					\
      -cos_z * sin_az - sin_z * sin_el * cos_az,	\
      -sin_z * cos_az - cos_z * sin_el * sin_az,	\
      cos_z * cos_el,					\
      sin_z * sin_az - cos_z * sin_el * cos_az,		\
      cos_el * sin_az,					\
      sin_el,						\
      cos_el * cos_az					\
      );						\
  MATVEC_END



/**
 * Set a 3x3 (sub)matrix to a rotation matrix representing rotation
 * of the <em>coordinate system</em> through a ZXZ Euler angle
 * rotation sequence.
 *
 *
 * \hideinitializer
 *
 * @param matrix    result 3x3 (sub)matrix extractor
 *
 * @param phi	    The 1st Euler angle (in radians).
 *                  This is a rotation about the current Z axis.
 *                  Positive phi takes Y towards X.
 *
 * @param theta     The 2nd Euler angle (in radians)
 *                  This is a rotation about the current X axis.
 *                  Positive theta takes Z towards Y.

 * @param psi       The 3rd Euler angle (in radians)
 *                  This is a rotation about the current Z axis(again).
 *                  Positive psi takes Y towards X.
 *
 * See also <a href="http://mathworld.wolfram.com/EulerAngles.html">mathworld</a>
 */
 /* note: filled in by row */

#define MATVEC_MAT3X3_ROT_ZXZ(					\
  matrix,phi,theta,psi						\
  )								\
  MATVEC_BEGIN							\
    double cos_phi   = cos( phi );				\
    double sin_phi   = sin( phi );				\
    double cos_theta = cos( theta  );				\
    double sin_theta = sin( theta  );				\
    double cos_psi   = cos( psi );				\
    double sin_psi   = sin( psi );				\
    MATVEC_MAT3X3_ASSIGN_COMP(					\
      matrix,							\
      cos_psi * cos_phi - cos_theta * sin_phi * sin_psi,	\
      cos_psi * sin_phi + cos_theta * cos_phi * sin_psi,	\
      sin_psi * sin_theta,					\
      -sin_psi * cos_phi - cos_theta * sin_phi * cos_psi,	\
      -sin_psi * sin_phi + cos_theta * cos_phi * cos_psi,	\
      cos_psi * sin_theta,					\
      sin_theta * sin_phi,					\
      -sin_theta * cos_phi,					\
      cos_theta							\
      );							\
    MATVEC_END

/** \} */


/*===========================================================================*/

/** \name Basic Vector Operations
 *
 */

/** \{ */


/** \cond */

/**
 * \internal
 * \hideinitializer
 *  Assign a vector to another
 *
 *
 * @param d0    the [0] component of the destination vector
 * @param d1    the [1] component of the destination vector
 * @param d2    the [2] component of the destination vector
 * @param s0    the [0] component of the source vector
 * @param s1    the [1] component of the source vector
 * @param s2    the [2] component of the source vector
 *
 * 
 */
#define MATVEC_VEC_ASSIGN_COMP(	\
  d0,d1,d2,			\
  s0,s1,s2			\
  )				\
  MATVEC_BEGIN			\
    (d0) = (s0);		\
    (d1) = (s1);		\
    (d2) = (s2);		\
  MATVEC_END

/** \endcond */

/**
 *  Assign a vector to another.
 *
 * \hideinitializer
 *
 * @param dst  the destination vector extractor
 * @param src  the source vector extractor
 *
 */

#define MATVEC_VEC_ASSIGN(dst,src)	\
  MATVEC_VEC_ASSIGN_COMP(dst,src)




/** \cond */

/**
 * \internal
 * \hideinitializer
 *  Multiply a vector by a scalar
 *
 * @param d0    the [0] component of the destination vector
 * @param d1    the [1] component of the destination vector
 * @param d2    the [2] component of the destination vector
 * @param s0    the [0] component of the source vector
 * @param s1    the [1] component of the source vector
 * @param s2    the [2] component of the source vector
 * @param fac   the scale factor
 *
 * 
 */
#define MATVEC_VEC_SCALE_COMP(		\
  d0,d1,d2,				\
  s0,s1,s2,				\
  fac					\
  )					\
  MATVEC_VEC_ASSIGN_COMP(		\
    (d0), (d1), (d2),			\
    (fac) * (s0),			\
    (fac) * (s1),			\
    (fac) * (s2) )


/** \endcond */


/**
 *  Scale a vector.
 *  The result vector may be the input vectors
 *
 * \hideinitializer
 *
 * @param res  the result vector extractor
 * @param vin  the input vector extractor
 * @param fac  the factor with which the vector will be multiplied
 *
 */

#define MATVEC_VEC_SCALE(res,vin,fac)		\
  MATVEC_VEC_SCALE_COMP(res,vin,fac)



/** \cond */

/**
 * \internal
 * \hideinitializer
 *  Add two vectors and store the result in another vector.
 *  The result vector may be one of the input vectors
 *
 *
 * @param d0    the [0] component of the result vector
 * @param d1    the [1] component of the result vector
 * @param d2    the [2] component of the result vector
 * @param a0    the [0] component of the first vector
 * @param a1    the [1] component of the first vector
 * @param a2    the [2] component of the first vector
 * @param b0    the [0] component of the second vector
 * @param b1    the [1] component of the second vector
 * @param b2    the [2] component of the second vector
 *
 * 
 */
#define MATVEC_VEC_ADD_COMP(	\
  d0,d1,d2,			\
  a0,a1,a2,			\
  b0,b1,b2			\
  )				\
  MATVEC_BEGIN			\
    d0 = (a0) + (b0);		\
    d1 = (a1) + (b1);		\
    d2 = (a2) + (b2);		\
  MATVEC_END

/** \endcond */

/**
 *  Add two vectors and store the result in another vector.
 *  The result vector may be one of the input vectors
 *
 * \hideinitializer
 *
 * @param sum   the result vector extractor
 * @param veca  the first vector extractor
 * @param vecb  the second vector extractor
 *
 */

#define MATVEC_VEC_ADD(sum,veca,vecb)	\
  MATVEC_VEC_ADD_COMP(sum,veca,vecb)

/** \cond */

/**
 * \internal
 * \hideinitializer
 *  Subtract two vectors and store the result in another vector.
 *  The result vector may be one of the input vectors
 *
 * @param d0    the [0] component of the result vector
 * @param d1    the [1] component of the result vector
 * @param d2    the [2] component of the result vector
 * @param a0    the [0] component of the first vector
 * @param a1    the [1] component of the first vector
 * @param a2    the [2] component of the first vector
 * @param b0    the [0] component of the second vector
 * @param b1    the [1] component of the second vector
 * @param b2    the [2] component of the second vector
 *
 * The point position will be updated to reflect the projection.
 */
#define MATVEC_VEC_SUB_COMP(	\
  d0,d1,d2,			\
  a0,a1,a2,			\
  b0,b1,b2			\
  )				\
  MATVEC_BEGIN			\
    d0 = (a0) - (b0);		\
    d1 = (a1) - (b1);		\
    d2 = (a2) - (b2);		\
  MATVEC_END

/** \endcond */

/**
 * Subtract two vectors.
 * The result vector may be one of the input vectors.
 *
 * \hideinitializer
 *
 * @param res   the result vector extractor
 * @param veca  the first vector extractor
 * @param vecb  the second vector extractor
 *
 */

#define MATVEC_VEC_SUB(res,veca,vecb)	\
  MATVEC_VEC_SUB_COMP(res,veca,vecb)


/** \cond */

/**
 * \internal
 * \hideinitializer
 *  
 * Project a point along a direction.  The result vector may be the
 * same as the position vector.
 *
 *
 * @param res0    the [0] component of the result
 * @param res1    the [1] component of the result
 * @param res2    the [2] component of the posult
 * @param pos0    the [0] component of the position.
 * @param pos1    the [1] component of the position.
 * @param pos2    the [2] component of the position.
 * @param dir0    the [0] component of the direction vector.
 * @param dir1    the [1] component of the direction vector.
 * @param dir2    the [2] component of the direction vector.
 *
 */
#define MATVEC_VEC_PROJECT_COMP(	\
  res0,res1,res2,			\
  pos0,pos1,pos2,			\
  dir0,dir1,dir2,			\
  s					\
  )					\
  MATVEC_BEGIN				\
    res0 = pos0 + (s) * (dir0);		\
    res1 = pos1 + (s) * (dir1);		\
    res2 = pos2 + (s) * (dir2);		\
  MATVEC_END

/** \endcond */

/**
 * Project a point along a vector.
 *
 * \hideinitializer
 *
 * @param res   the result vector extractor
 * @param pos   the point position vector extractor
 * @param dir   the direction vector extractor
 * @param s     the fractional distance along the direction vector to
 *              project.
 *
 * The result vector may be the same as the position vector.
 */

#define MATVEC_VEC_PROJECT(res,pos,dir,s)	\
  MATVEC_VEC_PROJECT_COMP(res,pos,dir,s)

/** \cond */
/**
 * This macro returns the dot product of two vectors.
 *
 * \internal
 * \hideinitializer
 *
 * @param v0    [0] component of first vector
 * @param v1    [1] component of first vector
 * @param v2    [2] component of first vector
 * @param w0    [0] component of second vector
 * @param w1    [1] component of second vector
 * @param w2    [2] component of second vector
 */
#define MATVEC_VEC_DOTPROD_COMP(v0,v1,v2,w0,w1,w2)	\
    (v0 * w0 + v1 * w1 + v2 * w2)		\

/** \endcond */

/**
 * This macro returns the dot product of two vectors.
 *
 * \hideinitializer
 *
 * @param v     1st vector extractor
 * @param w     2nd vector extractor
 */
#define MATVEC_VEC_DOTPROD(v,w)	\
    MATVEC_VEC_DOTPROD_COMP(v,w)

/** \cond */
/**
 * This macro returns the cross product of two vectors.
 *
 * \internal
 * \hideinitializer
 *
 * @param r0    [0] component of result vector
 * @param r1    [1] component of result vector
 * @param r2    [2] component of result vector
 * @param a0    [0] component of first vector
 * @param a1    [1] component of first vector
 * @param a2    [2] component of first vector
 * @param b0    [0] component of second vector
 * @param b1    [1] component of second vector
 * @param b2    [2] component of second vector
 */
#define MATVEC_VEC_CROSSPROD_COMP(	\
  r0,r1,r2,				\
  a0,a1,a2,				\
  b0,b1,b2				\
  )					\
  MATVEC_BEGIN				\
  (r0) = (a1) * (b2) - (b1) * (a2);	\
  (r1) = (a2) * (b0) - (b2) * (a0);	\
  (r2) = (a0) * (b1) - (b0) * (a1);	\
  MATVEC_END

/** \endcond */

/**
 * This macro returns the dot product of two vectors.
 *
 * \hideinitializer
 *
 * @param res   result vector extractor
 * @param v     1st vector extractor
 * @param w     2nd vector extractor
 */
#define MATVEC_VEC_CROSSPROD(res,v,w)		\
  MATVEC_VEC_CROSSPROD_COMP(res,v,w)


/** \cond */
/**
 * This macro normalizes a vector
 *
 * \internal
 * \hideinitializer
 *
 * @param v0    [0] component of vector
 * @param v1    [1] component of vector
 * @param v2    [2] component of vector
 */
#define MATVEC_VEC_NORM_COMP(v0,v1,v2)	\
  MATVEC_BEGIN					\
    double _norm = sqrt(MATVEC_VEC_DOTPROD_COMP(v0,v1,v2,v0,v1,v2));	\
    v0 /= _norm;				\
    v1 /= _norm;				\
    v2 /= _norm;				\
  MATVEC_END

/** \endcond */


/**
 * Normalize a vector
 *
 * \hideinitializer
 *
 * @param vector   the vector extractor
 */
#define MATVEC_VEC_NORM(vector) \
    MATVEC_VEC_NORM_COMP(vector)

/** \} */

/*===========================================================================*/

/** \name Vector Transformations
 *
 * These macros operate on vectors. Depending upon the convention used
 * when constructing the rotation matrices and translation vectors,
 * these operations will either transform the coordinate frames of the
 * vectors and specify the vector in the new coordinate frame, or will
 * transform the vector within the existing coordinate frame.  Be sure
 * to understand which convention was used, and don't mix 'em up!
 *
 */

/** \{ */

/** \cond */

/** 
 * \brief Apply a rotation followed by a translation to a vector.
 * 
 * Vectors are specified by their (many) components.  Use 
 * ::MATVEC_VEC_ROTATE_TRANSLATE instead!
 *
 * \hideinitializer
 *
 * @param r0      the [0] component of the result vector.
 * @param r1      the [1] component of the result vector.
 * @param r2      the [2] component of the result vector.
 * @param vin0    the [0] component of the input vector.
 * @param vin1    the [1] component of the input vector.
 * @param vin2    the [2] component of the input vector.
 * @param m00     the [0][0] component of the rotation matrix
 * @param m01     the [0][1] component of the rotation matrix
 * @param m02     the [0][2] component of the rotation matrix
 * @param m10     the [1][0] component of the rotation matrix
 * @param m11     the [1][1] component of the rotation matrix
 * @param m12     the [1][2] component of the rotation matrix
 * @param m20     the [2][0] component of the rotation matrix
 * @param m21     the [2][1] component of the rotation matrix
 * @param m22     the [2][2] component of the rotation matrix
 * @param vtr0    the [0] component of the translation vector.
 * @param vtr1    the [1] component of the translation vector.
 * @param vtr2    the [2] component of the translation vector.
 *
 * The parameters may be used more than once, so ensure that if they
 * are expressions that there are no side effects.  This macro is
 * designed to be used with the ::MATVEC_ROTATE_TRANSLATE macro,
 * not on its own.
 *
 */

#define MATVEC_VEC_ROTATE_TRANSLATE_COMP(				\
  r0,   r1,   r2,							\
  vin0, vin1, vin2,							\
  m00,  m01,  m02,							\
  m10,  m11,  m12,							\
  m20,  m21,  m22,							\
  vtr0, vtr1, vtr2							\
  )									\
  MATVEC_BEGIN								\
  r0									\
    = (m00) * (vin0)							\
    + (m01) * (vin1)							\
    + (m02) * (vin2) + (vtr0);						\
  r1									\
    = (m10) * (vin0)							\
    + (m11) * (vin1)							\
    + (m12) * (vin2) + (vtr1);						\
  r2									\
    = (m20) * (vin0)							\
    + (m21) * (vin1)							\
    + (m22) * (vin2) + (vtr2);						\
  MATVEC_END


/** \endcond */


/** 
 * \brief Apply a rotation followed by a translation to a vector.
 *
 * \hideinitializer
 *
 * @param res     the result vector extractor
 * @param vin     the initial vector extractor
 * @param matrix  the rotation matrix extractor
 * @param vtrans  the translation vector extractor
 *
 *
 * The parameters may be used more than once, so ensure that if they
 * are expressions that there are no side effects.
 *
 * Example:
 *
 * The convention is to use homogenous coordinates and the
 * transformation is stored as a 4x4 C multidimensional array.  In
 * this case, the rotation matrix is stored in the upper 3x3 submatrix
 * and the translation vector is stored in the right-most column.
 * The input and output vectors are stored in 3 element C arrays.

 * \code

 double transform[4][4];
 double vin[3], res[3];

 MATVEC_VEC_ROTATE_TRANSLATE(
			      MATVEC_VEC_AS_ARRAY(res,1),
			      MATVEC_VEC_AS_ARRAY(vin,1),
			      MATVEC_MAT3X3_AS_MDARRAY(transform),
 			      MATVEC_VEC_AS_ARRAY(&transform[0][3], 4)
			      );
 * \endcode
 *
 * Example:
 *
 * The rotation matrix is stored in a 3x3 C multidimensional array and
 * the translation vector is stored in a 3 element C array.  The input
 * and output vectors are stored in 3 element C arrays.
 *
 * \code

 double transform[3][3];
 double vtrans[3], vin[3], vout[3];

 MATVEC_VEC_ROTATE_TRANSLATE(
			       MATVEC_VEC_AS_ARRAY(res,1),
			       MATVEC_VEC_AS_ARRAY(vin,1),
 			       MATVEC_MAT3X3_AS_MDARRAY(transform),
			       MATVEC_VEC_AS_ARRAY(vtrans,1)
			       );
 * \endcode
 *
 */
#define MATVEC_VEC_ROTATE_TRANSLATE(res,vin,matrix,vtrans)	\
  MATVEC_VEC_ROTATE_TRANSLATE_COMP(res,vin,matrix,vtrans) 

/*===========================================================================*/

/** \cond */

/** 
 * \brief Apply a translation followed by a rotation to a vector.
 * \hideinitializer
 *
 * Vectors are specified by their (many) components.  Use 
 * ::MATVEC_VEC_TRANSLATE_ROTATE instead!
 *
 * @param ctype	  the C type of an array component
 * @param res0    the [2] component of the output vector.
 * @param res1    the [1] component of the output vector.
 * @param res2    the [2] component of the output vector.
 * @param vin0    the [0] component of the input vector.
 * @param vin1    the [1] component of the input vector.
 * @param vin2    the [2] component of the input vector.
 * @param vtr0    the [0] component of the translation vector.
 * @param vtr1    the [0] component of the translation vector.
 * @param vtr2    the [1] component of the translation vector.
 * @param m00     the [0][0] component of the rotation matrix
 * @param m01     the [0][1] component of the rotation matrix
 * @param m02     the [0][2] component of the rotation matrix
 * @param m10     the [1][0] component of the rotation matrix
 * @param m11     the [1][1] component of the rotation matrix
 * @param m12     the [1][2] component of the rotation matrix
 * @param m20     the [2][0] component of the rotation matrix
 * @param m21     the [2][1] component of the rotation matrix
 * @param m22     the [2][2] component of the rotation matrix
 *
 * The parameters may be used more than once, so ensure that if they
 * are expressions that there are no side effects.  This macro is
 * designed to be used with the ::MATVEC_TRANSLATE_ROTATE macro, not
 * on its own.
 */

#define MATVEC_VEC_TRANSLATE_ROTATE_COMP(	\
  ctype,					\
  res0, res1, res2,				\
  vin0, vin1, vin2,				\
  vtr0, vtr1, vtr2,				\
  m00, m01, m02,				\
  m10, m11, m12,				\
  m20, m21, m22					\
  )						\
  MATVEC_BEGIN					\
  ctype _v0 = (vin0) + (vtr0);			\
  ctype _v1 = (vin1) + (vtr1);			\
  ctype _v2 = (vin2) + (vtr2);			\
  (res0)					\
    = (m00) * _v0				\
    + (m01) * _v1				\
    + (m02) * _v2;				\
  (res1)					\
    = (m10) * _v0				\
    + (m11) * _v1				\
    + (m12) * _v2;				\
  (res2)					\
    = (m20) * _v0				\
    + (m21) * _v1				\
    + (m22) * _v2;				\
  MATVEC_END


/** \endcond */

/** 
 * \brief Apply a translation followed by a rotation to a vector.
 *
 * \hideinitializer
 *
 * @param type	  the C type of a vector component (e.g. double)
 * @param vout    the output vector extractor
 * @param vin     the input vector extractor
 * @param vtrans  the translation vector extractor
 * @param matrix  the rotation matrix extractor
 *
 * The parameters may be used more than once, so ensure that if they
 * are expressions that there are no side effects.
 *
 * Example:
 *
 * The convention is to use homogenous coordinates and the
 * transformation is stored as a 4x4 C multidimensional array.  In
 * this case, the rotation matrix is stored in the upper 3x3 submatrix
 * and the translation vector is stored in the right-most column.
 * The input and output vectors are stored in 3 element C arrays.

 * \code

 double transform[4][4];
 double vin[3], vout[3];

 MATVEC_VEC_TRANSLATE_ROTATE( double,
			      MATVEC_VEC_AS_ARRAY(vout,1),
			      MATVEC_VEC_AS_ARRAY(vin,1),
			      MATVEC_VEC_AS_ARRAY(&transform[0][3], 4),
                              MATVEC_MAT3X3_AS_MDARRAY(transform)
			     );
 * \endcode
 *
 *
 * Example:
 *
 * The rotation matrix is stored in a 3x3 C multidimensional array and
 * the translation vector is stored in a 3 element C array.  The input
 * and output vectors are stored in 3 element C arrays.
 *
 * \code

 double transform[3][3];
 double vtrans[3], vin[3], vout[3];

 MATVEC_VEC_TRANSLATE_ROTATE( double,
			      MATVEC_VEC_AS_ARRAY(vout,1),
			      MATVEC_VEC_AS_ARRAY(vin,1),
			      MATVEC_VEC_AS_ARRAY(vtrans,1),
                              MATVEC_MAT3X3_AS_MDARRAY(transform)
			      );
 * \endcode
 */

#define MATVEC_VEC_TRANSLATE_ROTATE(type,vout,vin,vtrans,matrix) \
  MATVEC_VEC_TRANSLATE_ROTATE_COMP(type,vout,vin,vtrans,matrix)

/** \} */


#endif /* MATVEC_MACROS_H */
