#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <bpipe/bpipe.h>
#include <bpipe/datatypes.h>

/* simple print error and exit routine */
#define error(string)							\
  do {									\
      fprintf( stderr, __FILE__ " %d: %s\n", __LINE__, string );	\
      exit( EXIT_FAILURE );						\
     } while (0)

/* -----------------------------------------------------------------------*/

/* forward definitions */
void *dpktf_data( BPipe *bpipe, char *name, void *core_image,
		  BPDataType type );

/* -----------------------------------------------------------------------*/

int
main (int argc, char *argv[])
{
  BPipe *bpipe;
  BPipeOutput *bpo;
  BPMatrix *matrix;
  void *core;

  unsigned int nspokes = 200;
  unsigned int nspokes_new = 13;
  unsigned int nrings = 200;

  DVector3 *position, *direction;
  double *polarization;
  double *graze_angle;
  double *weight;
  int stuff_init;
  size_t i;

  size_t nphots = 1;

  if ( argc > 1 )
    nphots = atoi( argv[1] ) || 1;

  if ( NULL == ( bpipe = bpipe_new( ) ) )
    error( bpipe_strerror( bpipe_errno ) );

  /* ------------------------------------------------------------ */

  /*
     create new header fields. note that none of the following
     calls to bpipe_hdrf_add has the copy flag set, as there's no
     need to duplicate this data
   */

  if ( bpipe_hdrf_add( bpipe, "nrings", BPDType_uint, NULL, &nrings, 0) )
    error( bpipe_strerror( bpipe_errno ) );

  if ( bpipe_hdrf_add( bpipe, "nspokes", BPDType_uint, NULL, &nspokes, 0) )
    error( bpipe_strerror( bpipe_errno ) );

  if ( bpipe_hdrf_string_add( bpipe, "comment", "first comment",
			      (size_t) 0, (size_t) 0 ) )
    error( bpipe_strerror( bpipe_errno ) );

  if ( bpipe_hdrf_string_add( bpipe, "comment", "second comment",
			      (size_t) 0, (size_t) 0 ) )
    error( bpipe_strerror( bpipe_errno ) );

  /* ------------------------------------------------------------ */

  /* create new data packet fields */

  if ( bpipe_dpktf_add( bpipe,	"position",	BPDType_DVector3, NULL ) )
    error( bpipe_strerror( bpipe_errno ) );

  if ( bpipe_dpktf_add( bpipe,	"direction",	BPDType_DVector3, NULL ) )
    error( bpipe_strerror( bpipe_errno ) );

  if ( bpipe_dpktf_add( bpipe,	"graze_angle",	BPDType_double,   NULL ) )
    error( bpipe_strerror( bpipe_errno ) );

  if ( bpipe_dpktf_add( bpipe,	"weight",	BPDType_double,   NULL ) )
    error( bpipe_strerror( bpipe_errno ) );

  /* create a one dimensional array with 4 elements */
  if ( bpipe_dpktf_array_add( bpipe, "polarization", BPDType_double,
			      (size_t) 4 ) )
    error( bpipe_strerror( bpipe_errno ) );

  /*
     create a 3x3x3 cube. since the bpipe library takes responsibility for
     the matrix after a bpipe_dpktf_add, don't need to delete the matrix
   */
  if ( NULL == (matrix = bpipe_matrix_new_va( (size_t) 3, 3, 3, 3 )) ||
       bpipe_dpktf_add( bpipe, "stuff", BPDType_int, matrix ) )
    error( bpipe_strerror( bpipe_errno ) );

  /* ------------------------------------------------------------ */

  /* resize nspokes array for the fun of it */

  if ( NULL ==
       ( matrix = bpipe_hdrf_matrix( bpipe, "nspokes", BPHdrfIdx_LAST )) )
    error( bpipe_strerror( bpipe_errno ) );

  matrix->extent[0] = 5;

  /*
    initialize it with the contents of nspokes_new (set in the
    definitions above)
   */
  if ( bpipe_hdrf_resize( bpipe, "nspokes", BPHdrfIdx_LAST, matrix,
		       NULL, NULL, NULL, &nspokes_new, (size_t) 1 ) )
    error( bpipe_strerror( bpipe_errno ) );

  /* ------------------------------------------------------------ */

  /* open up output stream to stdout */
  if ( NULL == ( bpo = bpipe_output( bpipe, "stdout" ) ) )
    error( bpipe_strerror( bpipe_errno ) );

  /* map data packet fields */
  if ( NULL == ( core = bpipe_map_alloc( bpipe, 1, NULL) ) &&
       bpipe_errno != BPNOERROR )
    error( bpipe_strerror( bpipe_errno ) );

  /* output header. must do this after calling bpipe_map */
  if ( bpipe_write_hdr(bpipe) )
    error( bpipe_strerror( bpipe_errno ) );

  /*
    get positions of fields in data packet. dpktf_data is *not* a
    bpipe library routine
   */
  position     = (DVector3 *) dpktf_data( bpipe, "position",     core, BPDType_DVector3 );
  direction    = (DVector3 *) dpktf_data( bpipe, "direction",    core, BPDType_DVector3 );
  weight       = (double *)   dpktf_data( bpipe, "weight",       core, BPDType_double );
  graze_angle  = (double *)   dpktf_data( bpipe, "graze_angle",  core, BPDType_double );
  polarization = (double *)   dpktf_data( bpipe, "polarization", core, BPDType_double );

  /*
     use the fact that there's only one data packet core image buffer
     to preset some stuff
   */
  *weight = 235.3;
  *graze_angle = 1;

  for ( i = 0 ; i < 4 ; i++ )
    polarization[i] = i;

  stuff_init = 42;
  bpipe_dpktf_init( bpipe_dpktf( bpipe, "stuff" ), core, &stuff_init );

  /* create the data packets */
  for (i = 0; i < nphots ; i++)
  {
    /* play with the information */
    position->x = position->y = position->z = i;

    direction->x = direction->y = direction->z = i+2;

    /* write the data packet out */
    if ( bpipe_write_dpkt( bpipe, core, bpo ) )
      error( "error writing photon" );
  }

  bpipe_delete(bpipe);

  return EXIT_SUCCESS;
}

/* simple routine to get the data pointer for a named data packet field */
void *
dpktf_data( BPipe *bpipe, char *name, void *core_image, BPDataType type )
{
  void *data = bpipe_dpktf_datap( bpipe, core_image, name, type );

  if ( NULL == data )
    error( name );

  return data;
}
