#ifndef BPIPE_COMMON_H
#define BPIPE_COMMON_H

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2005-2011 Smithsonian Astrophysical Observatory
 *
 * This file is part of bpipe
 *
 * bpipe is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * -->8-->8-->8-->8-- */

#include <stddef.h>
#include <stdio.h>
#include <limits.h>

#ifdef __cplusplus
extern "C" {
#endif

/*
   enums for intrinsic data types.  these must be in the
   same order as the bpipe_datatypes table.
*/
typedef enum
{
 _BPDType_NOTYPE,
  BPDType_char,
  BPDType_double,
  BPDType_int,
  BPDType_uint,
  BPDType_DVector2,
  BPDType_DVector3,
  BPDType_IVector2,
  BPDType_IVector3,
  BPDType_UIVector2,
  BPDType_UIVector3,
  BPDType_DComplex,
  BPDType_DCVector2,
  BPDType_DCVector3,
 _BPDType_num
} BPDataType;
#define BPDType_NOTYPE _BPDType_NOTYPE
#define BPDType_num (_BPDType_num-1)

/* identifiers for where a field specification/operation is applicable */
typedef enum
{
  BPDSite_INPUT,
  BPDSite_CORE,
  BPDSite_OUTPUT
} BPDataSite;

/* n-dimensional matrix */
typedef struct
{
  size_t nd;		/* number of dimensions */
  size_t *extent;	/* nd length array with extents */
  size_t ne;		/* the total number of elements */
} BPMatrix;

typedef enum
{
  BPXMap_Input_to_Core,
  BPXMap_Core_to_Output,
  BPXMap_Core_to_Core
} BPXMap;

/* internal information.  unfortunately must reveal it to the prying
   eyes of those for whom the documentation just isn't good enough,
   is it? */
struct  DpktFieldPrivate
{
  char *name;		/* field name */
  size_t index;		/* field index.  used during field size sort to keep
  			   input fields together */
  BPDataType type;	/* field data type */

  BPipe *bpipe;	/* parent bpipe structure */

  BPMatrix *input;	/* storage description of input state */
  BPMatrix *core;	/* storage description of core state */
  BPMatrix *output;	/* storage description of default output state */

  BPMatrixMap *input_to_core;	/* map from input to core states */
  BPMatrixMap *core_to_output;	/* default map from core to output states */

  size_t input_offset;	/* offset into input object */
  size_t core_offset;	/* offset into core object */

  int global_delete;	/* flag indicating the field is globally
			   deleted on output */
  int output_delete;	/* flag indicating that it is deleted for some
			   output channel */
};

/* macros to access data packet field data */
#define bpipe_dpktf_val(dpktf,dpkt,type) \
   (*((type *) ((char *)(dpkt) + ((struct DpktFieldPrivate*)dpktf)->core_offset)))

#define bpipe_dpktf_valn(dpktf,dpkt,type,n) \
   (((type *) ((char *)(dpkt) + ((struct DpktFieldPrivate*)dpktf)->core_offset))[n])

#define bpipe_dpktf_arr(dpktf,dpkt,type) \
    ((type *) ((char *)(dpkt) + ((struct DpktFieldPrivate*)dpktf)->core_offset))

#define bpipe_dpktf_data(dpktf,dpkt) \
    ((char *)(dpkt) + ((struct DpktFieldPrivate*)dpktf)->core_offset)

#define bpipe_dpktf_offset(dpktf) \
    ((struct DpktFieldPrivate*)dpktf)->core_offset

/* access header parameter duplicates */
#define BPHdrfIdx_LAST	((size_t) INT_MAX)
#define BPHdrfIdx_ALL	((size_t) INT_MAX - 1)

/* access all output channels */
#define BPOutputChannel_ALL NULL

typedef enum
{
  BPNOERROR,
  BPEBADARG,
  BPENOMEM,
  BPEBADPIPE,
  BPEIOERR,
  BPENOFILE,

  _BPEnum

} BPerrno;

/* bpipe managed/created  header field info */
#define BPHN_n_dpkts		"_n_dpkts"
#define BPHT_n_dpkts		BPDType_uint

#define BPHN_n_dpkts_valid	"_n_dpkts_valid"
#define BPHT_n_dpkts_valid	BPDType_int

#ifdef __cplusplus
}
#endif

#endif /* BPIPE_COMMON_H */
