#ifndef rl_RayMath_h_INCLUDED
#define rl_RayMath_h_INCLUDED

// File:   rl_RayMath.h
// Author: Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of rl_ray
 *
 * rl_ray is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * rl_ray is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

/****************************************************************************
 * Description: collection of methods for rl_RayMath
 *
 * History
 *--------
 * 1.0.1 2004-Dec-07  tjg  simplify/remove include guards
 * 0.0.0 1998-Jan-30  tjg  original version
 */

#include <dvm3/dvm3_rotmat.h>                   // dvm3_RotMat

//########################################################################
// rl_RayMath
//########################################################################
/** 
 * \class rl_RayMath
 *
 * A class gathering together static methods for operations on
 * ray position or direction vectors.
 *
 *  The class is a simple class to handle common numerical
 *  operations on 3-vectors of floating point T_fps representing ray
 *  position vectors and ray direction vectors.
 *
 *  rl_RayMath has only static member functions; there are no
 *  member data.  Where possible, the static member functions are
 *  inlined.
 *
 *  The static member functions facilitate transformations between
 *  an initial coordinate system (termed "Standard" or STD) and
 *  another coordinate system (termed "Body Centered" or BCS).  This 
 *  is just convensient labeling which reflects a common usage of the
 *  transformations.
 *
 *  The BCS coordinate origin is (optionally) translated relative
 *  to the STD origin, and the orientation of the BCS coordinate axes
 *  relative to the STD axes is described by a proper rotation matrix.
 *
 *  The transformation from STD to BCS is accomplished by
 *  "rotate_translate" (or "rotate" if the coordinate origins are
 *  coincident).
 *
 *  The transformation from BCS back to STD is accomplished by
 *  "detranslate_derotate" (or "derotate" if the coordinate origins are
 *  coincident).
 *
 *  Direction vectors are always transformed using the "rotate"-"derotate"
 *  pair, NEVER the versions involving translation.
 *
 *  Positional vectors are transformed using "rotate_translate" and
 *  "detranslate_derotate"; if the STD and BCS coordinate systems origins
 *  are coincident, the transformation can be accomplished more economically
 *  using the "rotate" and "derotate" pair.
 *
 *  NOTE:  although dvm3_Matrix works with either dvm3_Vector's
 *  or plain old c-style one-dimensional arrays, any call to an 
 *  rl_RayMath method must use only one vector type within a
 *  given method call.
 */

class rl_RayMath
{
public:

  /**
   * Rotate vector from STD coordinates to BCS coordinates.
   *
   * @param rotmat rotation vector (STD to BCS) to be applied.
   * @param vector vector to be rotated.
   */
  inline static void rotate( dvm3_RotMat const& rotmat,
                             dvm3_Vector&       vector );

  /**
   * Rotate vector from STD coordinates to BCS coordinates.
   *
   * @param rotmat rotation vector (STD to BCS) to be applied.
   * @param vector c-style vector to be rotated.
   */
  inline static void rotate( dvm3_RotMat const& rotmat,
                             double             vector[] );

  /**
   * De-rotate vector from BCS coordinates back to STD coordinates.
   *
   * @param rotmat rotation vector (STD to BCS) to be used.  The
   *     \em INVERSE of rotmat is applied to the vector.
   * @param vector vector to be rotated.
   */
  inline static void derotate( dvm3_RotMat const& rotmat,
                               dvm3_Vector&       vector );

  /**
   * De-rotate vector from BCS coordinates back to STD coordinates.
   *
   * @param rotmat rotation vector (STD to BCS) to be used.  The
   *     \em INVERSE of rotmat is applied to the vector.
   * @param vector c-style vector to be rotated.
   */
  inline static void derotate( dvm3_RotMat const& rotmat,
                               double             vector[] );

  /**
   * Translate to BCS origin, then rotate from STD to BCS coordinates.
   *
   * @param trans  translation vector.
   * @param rotmat rotation vector (STD to BCS) to be applied.
   * @param vector vector to be derotated.
   */
  inline static void translate_rotate( dvm3_Vector const& trans,
                                       dvm3_RotMat const& rotmat,
                                       dvm3_Vector&       vector );

  /**
   * Translate to BCS origin, then rotate from STD to BCS coordinates.
   *
   * @param trans  translation vector.
   * @param rotmat rotation vector (STD to BCS) to be applied.
   * @param vector c-style vector to be derotated.
   */
  inline static void translate_rotate( double      const  trans[],
                                       dvm3_RotMat const& rotmat,
                                       double             vector[] );

  /**
   * Rotate from BCS to STD coordinates, then translate back to original 
   * position.  This is the inverse operation to translate_rotate.
   *
   * @param trans  translation vector.
   * @param rotmat rotation vector (STD to BCS) to be applied.
   * @param vector c-style vector to be derotated.
   */
  inline static void derotate_detranslate( dvm3_Vector const& trans, 
                                           dvm3_RotMat const& rotmat,
                                           dvm3_Vector&       vector );
  /**
   * Rotate from BCS to STD coordinates, then translate back to original 
   * position.  This is the inverse operation to translate_rotate.
   *
   * @param trans  c-style translation vector.
   * @param rotmat rotation vector (STD to BCS) to be applied.
   * @param vector c-style vector to be derotated.
   */
  inline static void derotate_detranslate( double      const  trans[],
                                           dvm3_RotMat const& rotmat,
                                           double             vector[] );
};

//########################################################################
//########################################################################
//
//    #    #    #  #          #    #    #  ######   ####
//    #    ##   #  #          #    ##   #  #       #
//    #    # #  #  #          #    # #  #  #####    ####
//    #    #  # #  #          #    #  # #  #            #
//    #    #   ##  #          #    #   ##  #       #    #
//    #    #    #  ######     #    #    #  ######   ####
//
//########################################################################
//########################################################################

//=========================================================================
// rotate from STD to BCS coordinates

//-------------------------------------------------------------------------
inline void rl_RayMath::
rotate( dvm3_RotMat const& rotmat,
        dvm3_Vector&       vector )
{
  dvm3_Vector tmp( vector );
  rotmat.mvmult( vector, tmp );
}

//-------------------------------------------------------------------------
inline void rl_RayMath::
rotate( dvm3_RotMat const& rotmat,
        double             vector[] )
{
  double tmp[3];  
  vm_V3Math<double>::copy(tmp, vector); 
  rotmat.mvmult( vector, tmp );
}

//=========================================================================
// derotate from BCS back to STD coordinates

//-------------------------------------------------------------------------
inline void rl_RayMath::
derotate( dvm3_RotMat const& rotmat,
          dvm3_Vector&       vector )
{
  dvm3_Vector tmp( vector );
  rotmat.mtvmult( vector, tmp );
}

//-------------------------------------------------------------------------
inline void rl_RayMath::
derotate( dvm3_RotMat const& rotmat,
          double             vector[] )
{
  double tmp[3]; 
  vm_V3Math<double>::copy(tmp, vector);
  rotmat.mtvmult( vector, tmp );
}

//=========================================================================
// translate to to BCS origin; rotate from STD to BCS coordinates

//-------------------------------------------------------------------------
inline void rl_RayMath::
translate_rotate( dvm3_Vector const& trans,
                  dvm3_RotMat const& rotmat,
                  dvm3_Vector&       vector )
{
  vector += trans;
  dvm3_Vector tmp( vector );
  rotmat.mvmult( vector, tmp );
}

//-------------------------------------------------------------------------
inline void rl_RayMath::
translate_rotate( double      const  trans[],
                  dvm3_RotMat const& rotmat,
                  double             vector[] )
{
  vm_V3Math<double>::add_eq( vector, trans );
  double tmp[3];
  vm_V3Math<double>::copy(tmp, vector);
  rotmat.mvmult( vector, tmp );
}

//=========================================================================
// derotate from BCS back to STD coordinates; detranslate back to STD origin

//-------------------------------------------------------------------------
inline void rl_RayMath::
derotate_detranslate( dvm3_Vector const& trans,
                      dvm3_RotMat const& rotmat,
                      dvm3_Vector&       vector )
{
  dvm3_Vector tmp( vector );
  rotmat.mtvmult( vector, tmp );
  vector -= trans;
}

//-------------------------------------------------------------------------
inline void rl_RayMath::
derotate_detranslate( double      const  trans[],
                      dvm3_RotMat const& rotmat,
                      double             vector[] )
{
  double tmp[3];
  vm_V3Math<double>::copy(tmp, vector);
  rotmat.mtvmult( vector, tmp );
  vm_V3Math<double>::sub_eq( vector, trans );
}

// rl_RayMath.h
#endif
