#ifndef rl_BasicRay_h_INCLUDED
#define rl_BasicRay_h_INCLUDED

// File:   rl_BasicRay.h
// Author: Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of rl_ray
 *
 * rl_ray is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * rl_ray is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <dvm3/dvm3.h>
#include <rl_basicray/rl_RayMath.h>
#include <Exception/Exception.h>

//########################################################################
// rl_BasicRay
//########################################################################
//

/** 
 * \class rl_BasicRay
 *
 * A basic ray:  encapsulates position and direction vectors, energy, 
 * and ray id.
 */

class rl_BasicRay
{
protected:

  /// ray position
  dvm3_Vector pos_;
  /// ray direction vector (direction cosines)
  dvm3_Vector dir_;
  /// ray energy
  double      energy_;
  /// ray id number
  long int    id_;

  /**
   * Initialize rl_BasicRay with ray initial position and direction vector,
   * energy, and id number
   *
   * @param pos ray position vector
   * @param dir ray direction vector
   * @param energy ray energy
   * @param id ray id number
   */ 
  void init( dvm3_Vector const& pos, dvm3_Vector const& dir, 
             double energy, long int id );

public:

  /// A virtual do-nothing destructor.
  virtual ~rl_BasicRay();
    //

  /** 
   * Default constructor; constructs a ray in an INVALID state.
   * Use init to initialize the fields.
   */
  rl_BasicRay();

  /**
   * Constructor.
   * Construct an rl_BasicRay from ray initial position and direction vector,
   * energy, and id number.
   *
   * @param pos ray initial position.
   * @param dir    ray initial direction unit vector.
   * @param energy ray energy, in keV.
   * @param id     ray id.
   */
  rl_BasicRay( dvm3_Vector const& pos, dvm3_Vector const& dir, 
               double energy, long int id );

  /** 
   * Copy constructor
   *
   * @param other ray to be copied.
   */
  rl_BasicRay( rl_BasicRay const& other );

  /** 
   * Reset ray id
   *
   * @param id new id value.
   */
  void set_id( long int id );

  /** 
   * Read-only access to current ray id
   *
   * @return ray id number
   */
  long int id() const;

  /** 
   * Read/write access to current ray energy
   *
   * @return a reference to the ray energy
   */
  double& energy();

  /** 
   * Read-only access to current ray position
   *
   * @return the ray energy
   */
  double  energy() const;

  /** 
   * Read/write access to component i of current ray position
   *
   * @param i index.
   *
   * @return a reference to the i component of the position vector.
   */
  double& position( int i );

  /** 
   * Read-only access to component i of current ray position
   *
   * @param i index.
   *
   * @return the i component of the position vector.
   */
  double  position( int i ) const;

  /** 
   * Read/write access to current ray position
   *
   * @return a reference to the position vector dvm3_Vector.
   */
  dvm3_Vector& position();

  /** 
   * Read-only access to current ray position
   *
   * @return a const reference to the position vector dvm3_Vector
   */
  dvm3_Vector const& position() const;

  /** 
   * Read/write access to component i of current ray direction 
   *
   * @param i index.
   */
  double& direction( int i );

  /** 
   * Read-only access to component i of current ray direction
   *
   * @param i index.
   *
   * @return component i of the position vector.
   */
  double direction( int i ) const;

  /** 
   * Read/write access to current ray direction vector
   *
   * @return reference to current ray direction vector dvm3_Vector
   */
  dvm3_Vector& direction();

  /** 
   * Read-only access to current ray direction vector
   *
   * @return const reference to current ray direction vector dvm3_Vector
   */
  dvm3_Vector const& direction() const;

  /** 
   * Reflect this ray's direction vector about a (surface) normal
   *
   * @param normal - surface normal unit vector to be used in reflecting
   *    this rl_BasicRay.
   *
   * @return sine of the graze angle between the ray and the surface
   */
  double reflect_direction_vector( dvm3_Vector const& normal );

  /** 
   * Project a distance s along this ray
   *
   * @param s distance to project the ray.
   */
  void project( double s );

  /** 
   * VIRTUAL: Translate to the BCS origin; rotate from STD to BCS coordinates
   *
   * @param trans  translation vector.
   * @param rotmat rotation matrix to be applied.  rotmat specifies
   *        a rotation from std to bcs.
   */
  virtual
  void translate_rotate( dvm3_Vector const& trans, 
                         dvm3_RotMat const& rotmat );

  /** 
   * VIRTUAL: Derotate back to STD coordinates; detranslate back to STD origin
   *
   * @param trans  translation vector.
   * @param rotmat rotation matrix to be applied.  rotmat specifies
   *        a rotation from std to bcs; the inverse of rotmat is
   *        applied to the ray.
   */
  virtual
  void derotate_detranslate( dvm3_Vector const& trans, 
                             dvm3_RotMat const& rotmat );

  /** 
   * Print the ray properties to an output stream.
   *
   * @param os      output stream.
   * @param prefix  an optional prefix string.
   * @param postfix an optional postfix string.
   *
   * @return output stream
   */
  std::ostream& print_on( std::ostream& os, char const prefix[] = "", 
                                            char const postfix[] = "" ) const;

  /** 
   * Print the ray properties to an output stream.
   *
   * @param os      output stream.
   *
   * @return output stream
   */
  friend std::ostream& operator<<( std::ostream& os, rl_BasicRay const& );
};

//########################################################################
//########################################################################
//
//    #    #    #  #          #    #    #  ######   ####
//    #    ##   #  #          #    ##   #  #       #
//    #    # #  #  #          #    # #  #  #####    ####
//    #    #  # #  #          #    #  # #  #            #
//    #    #   ##  #          #    #   ##  #       #    #
//    #    #    #  ######     #    #    #  ######   ####
//
//########################################################################
//########################################################################

//=========================================================================
// dtor, ctors, initializers...

//-------------------------------------------------------------------------
inline rl_BasicRay::
rl_BasicRay()
{}

//-------------------------------------------------------------------------
inline rl_BasicRay::
rl_BasicRay( dvm3_Vector const& pos, dvm3_Vector const& dir,
             double energy, long int id )
  : pos_(pos), dir_(dir), energy_(energy), id_(id)
{}

//-------------------------------------------------------------------------
inline rl_BasicRay::
rl_BasicRay( rl_BasicRay const& other )
  : pos_(other.pos_), dir_(other.dir_), energy_(other.energy_), id_(other.id_)
{}

//-------------------------------------------------------------------------
inline void rl_BasicRay::
init( dvm3_Vector const& pos, dvm3_Vector const& dir,
      double energy, long int id )
{
  pos_    = pos; 
  dir_    = dir; 
  energy_ = energy;
  id_     = id;
}

//=========================================================================
// accessors...

//-------------------------------------------------------------------------
inline long int rl_BasicRay::
id() const
{ return id_; }

inline void rl_BasicRay::
set_id( long int id )
{ id_ = id; }

//-------------------------------------------------------------------------
inline double& rl_BasicRay::
energy()
{ return energy_; }

//-------------------------------------------------------------------------
inline double rl_BasicRay::
energy() const
{ return energy_; }

//-------------------------------------------------------------------------
inline double& rl_BasicRay::
position( int i )
{ return pos_[i]; }

//-------------------------------------------------------------------------
inline double rl_BasicRay::
position( int i ) const
{ return pos_[i]; }

//-------------------------------------------------------------------------
inline dvm3_Vector& rl_BasicRay::
position()
{ return pos_; }

//-------------------------------------------------------------------------
inline dvm3_Vector const& rl_BasicRay::
position() const
{ return pos_; }

//-------------------------------------------------------------------------
inline double& rl_BasicRay::
direction( int i )
{ return dir_[i]; }

//-------------------------------------------------------------------------
inline double rl_BasicRay::
direction( int i ) const
{ return dir_[i]; }

//-------------------------------------------------------------------------
inline dvm3_Vector& rl_BasicRay::
direction()
{ return dir_; }

//-------------------------------------------------------------------------
inline dvm3_Vector const& rl_BasicRay::
direction() const
{ return dir_; }

//=========================================================================
// ray projection...

//-------------------------------------------------------------------------
inline
void rl_BasicRay::
project( double s )
{
  pos_ += s * dir_;
}

//=========================================================================
// translate to to bcs origin; rotate from std to bcs coordinates

//-------------------------------------------------------------------------
inline void rl_BasicRay::
translate_rotate( dvm3_Vector const& trans,
                  dvm3_RotMat const& rotmat )
{
  rl_RayMath::translate_rotate( trans, rotmat, pos_ );
  rl_RayMath::rotate(                  rotmat, dir_ );
}

//=========================================================================
// derotate from bcs back to std coordinates; detranslate back to std origin

//-------------------------------------------------------------------------
inline void rl_BasicRay::
derotate_detranslate( dvm3_Vector const& trans,
                      dvm3_RotMat const& rotmat )
{
  rl_RayMath::derotate_detranslate( trans, rotmat, pos_ );
  rl_RayMath::derotate(                    rotmat, dir_ );
}

// rl_BasicRay_h_INCLUDED
#endif
