#ifndef BSPLINE_EVAL_H
#define BSPLINE_EVAL_H

#include <stdio.h>

#include <BsplineEval/spline_coef.h>

typedef struct {

  double* t_knots; 
  /* The augumented t-knots, interior knots plus coincident end knots */

  double* z_knots;
  /* The augumented z-knots, interior knots plus coincident end knots */

  double *d, *difz, *dift;
  /* workspace for algorithm c */

  double* dt;
  /* workspace for algorithm c */

  double* dz;
  /* workspace for algorithm c */

} BsplineEvalSpline;


typedef struct {

  BsplineEvalSpline* eval_spline;
  spline_coef* scoef;
  char method1;
  char method2;

} BsplineInput;

typedef struct {

  double s                                                 /* s( theta, z ) */;
  double dsdz                                        /* ds( theta, z ) / dz */;
  double dsdt                                             /* ds( theta, t ) */;

} BsplineResult;

typedef enum {

  Bspline_Success                                               /* Success */,
  Bspline_OutOfRange                /* One of the arguments is out of range */,
  Bspline_OutOfRangeT
                 /* The theta value is not within the limits of the t_knots */,
  Bspline_OutOfRangeZ  /* The z value is outside the limits of the z_knots  */,
  Bspline_WrongMethod1  /* The method request must be one of : a, A, b or B */,
  Bspline_WrongMethod2   /* The method request must be one of : a, A, b or B */

} BsplineErr;

#ifdef __cplusplus
extern "C" {
#endif

BsplineErr
BsplineAlgo_a(
  double x           /* The point to evaluate the B-spline at */,
  size_t j_index     /* knot[ j_index-1 ] <= x < knot[ j_index ], j */,
  size_t order       /* the order of the B-spline, n */,
  double* c          /* the B-spline coefficients, subscripted c */,
  double* d          /* work space of dim : number of knots + order */,
  double* knot       /* the B-spline knots, subscripted x */,
  double* result     /* s(x) */
  );

BsplineErr
BsplineAlgo_b(
  double x           /* The point to evaluate the B-spline at */,
  size_t j_index     /* knot[ j_index-1 ] <= x < knot[ j_index ], j */,
  size_t order       /* the order of the B-spline, n */,
  double* c          /* the B-spline coefficients, subscripted c */,
  double* v          /* work space of dimension : number order + 1 */,
  double* knot       /* the B-spline knots, subscripted x */,
  double* result     /* s(x) */
  );

BsplineErr
BsplineAlgo_c(
  double theta       /* The value of theta to evaluate the B-spline */,
  double z           /* The value of z to evaluate the B-spline */,
  BsplineInput* bspline_input /*All neccessary B-spline input */,
  double* result     /* s( theta, z ) */
  );

BsplineErr
BsplineAlgo_c_derivs(
  double theta       /* The theta to evaluate the B-spline */,
  double z           /* The value of z to evaluate the B-spline */,
  BsplineInput* bspline_input    /* All B-spline input */,
  BsplineResult* bspline_result /* All B-spline output */
  );

void Bspline_err_msg( BsplineErr err, FILE* fp );

BsplineErr 
Bspline_determine_index(
  double* key             /* The key to search for */,
  const void* base              /* Points to the element at the base of array */,
  size_t num              /* Number of elements in the array */,
  double minval           /* The left most knot boundary */,
  double maxval           /* The right most knot boundary */,
  size_t* index           /* base[ index - 1 ] <= key < base[ index ] */
  );

BsplineErr
Bspline_eval(
  double theta            /* The value of theta to evaluate the B-spline */,
  double z                /* The value of z to evaluate the B-spline */,
  BsplineInput* bspline_input /*All neccessary B-spline input */,
  double* result          /* s( theta, z ) */
  );

BsplineErr
Bspline_eval_derivs(
  double theta            /* The theta to evaluate the B-spline */,
  double z                /* The value of z to evaluate the B-spline */,
  BsplineInput* bspline_input      /* All B-spline input */,
  BsplineResult* bspline_result /* All B-spline output */
  );

void Bspline_free( BsplineInput* ptr );

BsplineInput* Bspline_init( spline_coef* scoef );

#ifdef __cplusplus
}
#endif

#endif /* ! BSPLINE_EVAL_H */
