#ifndef BinaryPipe_H
#define BinaryPipe_H

// --8<--8<--8<--8<--
//
// Copyright (C) 2006 Smithsonian Astrophysical Observatory
//
// This file is part of BinaryPipe
//
// BinaryPipe is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// BinaryPipe is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the 
//       Free Software Foundation, Inc. 
//       51 Franklin Street, Fifth Floor
//       Boston, MA  02110-1301, USA
//
// -->8-->8-->8-->8--

#include <BinaryPipe/DataPacket.h>
#include <BinaryPipe/HeaderField.h>
#include <BinaryPipe/BinaryPipeException.h>

/**
   A simple binary pipe wrapper class to read and write binary pipe data.
   The user must supply a constructor of the form:
          Type( BPipe* bpipe, void* core );
   The constructor must assign the pointers to the data members within the
   user's class (Type).
 */
template <class Type>
class BinaryPipe {

  /// This operator outputs the contents of the object.
  friend std::ostream& operator << ( std::ostream& os, BinaryPipe<Type>& a ) {
    a.print( os );
    return os;
  }

  /// This operator outputs the contents of the object.
  friend std::ostream& operator << ( std::ostream& os, BinaryPipe<Type>* a ) {
    a->print( os );
    return os;
  }

public:

  ~BinaryPipe( );

  /// Default constructor
  BinaryPipe( ) : instream( NULL ), core( NULL ), outstream( NULL ),
    photon( NULL ) { }

  BinaryPipe( const char* input, const char* output=NULL,
	      HeaderField hdrf[]=NULL, DataPacket packet[]=NULL );

  /// Get the user's data.
  operator Type* ( ) { return photon; }

  /// To read the binary pipe.
  size_t read_dpkts( size_t n_pkts=1 ) const {
    return bpipe_read_dpkts( instream, core, n_pkts );
  }

  int write_dpkt( ) {
    if ( instream && outstream )
      return bpipe_write_dpkt( instream, core, outstream );
    else
      return BPEIOERR;
  }

  void init_BinaryPipe( const char* input, const char* output=NULL,
			HeaderField hdrf[]=NULL, DataPacket packet[]=NULL );

  void print( std::ostream& os ) const { os << photon; }

protected:

  BPipe *instream;
  void  *core;
  BPipeOutput *outstream;

  /**
     The user's data.
     The user must supply a constructor of the form: 
     Type( BPipe*, void* )
  */
  Type* photon;

};

#ifndef BinaryPipe_CC
#include <BinaryPipe/BinaryPipe.cc>
#endif

#endif
